/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions
 *   for Use of Fraunhofer Example Source Code", which is provided as a
 *   separate document together with this file. The "Terms & Conditions for
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the
 *   Example Source Code.
 *
\***************************************************************************/

#include <jni.h>
#include <android/log.h>

#include <stdlib.h>

#include "JniEnvGuard.h"
#include "AacEldDecoder.h"

#define LOG(...) __android_log_print(ANDROID_LOG_ERROR, "AacEldDecImpl", __VA_ARGS__)

// Internal decoder implementation
// hiding the JNI interaction from the rest
// of the native implementation
class AacEldDecoder::AacEldDecImpl {
public:
  // See comments in AacEldDecoder.h
  AacEldDecImpl(void *jvmHandle);
  ~AacEldDecImpl();

  bool configure(std::vector<unsigned char>& asc);
  bool decode(std::vector<unsigned char>& inAU, std::vector<unsigned char>& outSamples);
  void close();

private:
  // See comments in AacEldEncoder.c
  // Simply replace "encoder" with "decoder" there :-)
  JavaVM*   javavm;
  jclass    aacEldDecoderClass;
  jobject   aacEldDecoderInstance;
  jmethodID configureMethodId;
  jmethodID decodeMethodId;
  jmethodID closeMethodId;
  bool      jniInitialized;

  // Initializes the JNI members
  bool      initJni();
};

bool AacEldDecoder::AacEldDecImpl::initJni() {
  // Always get a pointer to the JNIEnv first
  JniEnvGuard env(javavm);

  // Try to find the java class abstracting the encoder
  // In the demo code this is de.fraunhofer.iis.aac_eld_encdec.AacEldDecoder
  // This could also be in the same package as your other Java classes are but
  // it can also be separate
  jclass decoderClass = env->FindClass("de/fraunhofer/iis/aac_eld_encdec/AacEldDecoder");

  // If we have found the class we store a reference to it
  // because when multiple native threads are using the native codec implementation
  // the Android Java Class loader can mess things up
  // (see http://developer.android.com/training/articles/perf-jni.html#faq_FindClass)
  // and fail to find the class. When this function is only one time called from the main
  // thread, chances are good that the reference to the class stays valid for the rest of the
  // application lifetime
  if (decoderClass && !aacEldDecoderClass) // Store a global reference for this application
    aacEldDecoderClass = reinterpret_cast<jclass>(env->NewGlobalRef(decoderClass));

  if (!decoderClass) { // in case of an error, first check if
    if (aacEldDecoderClass) // some thread got wild and we messed up the class loader stack
      decoderClass = aacEldDecoderClass; // try cached class if found before
    if(env->ExceptionCheck() == JNI_TRUE) { // and clear the pending exception that FindClass has already thrown
      env->ExceptionClear();
    }
    else { // all bets are off - cannot find class
      LOG("ERROR in AacEldDecoder::AacEldDecImpl::initJni(): FindClass failed!");
      jthrowable exc = env->ExceptionOccurred();
      if (exc) {
        LOG("######## JVM Execption follows: ###########");
        env->ExceptionDescribe();
        env->ExceptionClear();
        LOG("###########################################");
      }
      return false;
    }
  }

  // Once the class is found, the method ids of that class can be obtained
  // (use javap -s to print the signatures of Java class methods)
  jmethodID decoder_ctor   = env->GetMethodID(decoderClass, "<init>", "()V");
  configureMethodId        = env->GetMethodID(decoderClass, "configure", "([B)Z");
  decodeMethodId           = env->GetMethodID(decoderClass, "decode", "([B)[B");
  closeMethodId            = env->GetMethodID(decoderClass, "close", "()V");

  // It is an error if one of these is not found
  if (!decoder_ctor || !configureMethodId || !decodeMethodId || !closeMethodId) {
    return false;
  }

  // If all methods are found, create a new instance of the AacEldEncoder object
  jobject decoderObj              = env->NewObject(decoderClass, decoder_ctor);

  if (!decoderObj)
    return false;

  // Finally create a new global reference (otherwise the
  // just created object will be garbage collected as soon
  // as the current JNI call returns to Java)
  aacEldDecoderInstance = env->NewGlobalRef(decoderObj);

  if (!aacEldDecoderInstance)
    return false;

  return true;
}

AacEldDecoder::AacEldDecImpl::AacEldDecImpl(void *jvmHandle)
: javavm((JavaVM*)jvmHandle),
  aacEldDecoderClass(NULL),
  aacEldDecoderInstance(NULL),
  configureMethodId(NULL),
  decodeMethodId(NULL),
  closeMethodId(NULL),
  jniInitialized(false)
{
}

AacEldDecoder::AacEldDecImpl::~AacEldDecImpl() {

}

bool AacEldDecoder::AacEldDecImpl::configure(std::vector<unsigned char>& asc) {

  if (!jniInitialized)
    if (!initJni()) {
      return false;
    }

  // Configure Java Decoder
  JniEnvGuard env(javavm);

  jbyteArray inArray = env->NewByteArray(asc.size());
  jbyte *inBytes     = env->GetByteArrayElements(inArray, 0);
  memcpy(inBytes, &asc[0], sizeof(unsigned char)*asc.size());
  env->ReleaseByteArrayElements(inArray, inBytes, 0);

  jboolean result = env->CallBooleanMethod(aacEldDecoderInstance, configureMethodId, inArray);
  env->DeleteLocalRef(inArray);

  if (!result)
    return false;

  return true;
}

bool AacEldDecoder::AacEldDecImpl::decode(std::vector<unsigned char>& inAU, std::vector<unsigned char>& outSamples) {

  JniEnvGuard env(javavm);
  jbyteArray inArray = env->NewByteArray(inAU.size());
  jbyte     *inBytes = env->GetByteArrayElements(inArray, 0);

  memcpy(inBytes, &inAU[0], sizeof(unsigned char)*inAU.size());
  env->ReleaseByteArrayElements(inArray, inBytes, 0);

  jbyteArray resBuf = (jbyteArray) env->CallObjectMethod(aacEldDecoderInstance, decodeMethodId, inArray);
  env->DeleteLocalRef(inArray);

  if (!resBuf) {
    // Unfortunately, this is not always an error...
    memset(&outSamples[0], 0x00, outSamples.size());
    return true;
  }

  jsize resLen  = env->GetArrayLength(resBuf);
  jbyte *resByteBuf = env->GetByteArrayElements(resBuf, 0);
  if (resLen > outSamples.size()) {
    LOG("Decoder output too long...");
    env->ReleaseByteArrayElements(resBuf, resByteBuf, 0);
    env->DeleteLocalRef(resBuf);
    return false;
  }

  outSamples.resize(resLen);
  memcpy(&outSamples[0], resByteBuf, sizeof(unsigned char)*resLen);

  env->ReleaseByteArrayElements(resBuf, resByteBuf, 0);
  env->DeleteLocalRef(resBuf);

  return true;
}

void AacEldDecoder::AacEldDecImpl::close() {
  JniEnvGuard env(javavm);
  env->CallVoidMethod(aacEldDecoderInstance, closeMethodId);
  env->DeleteGlobalRef(aacEldDecoderInstance);
  aacEldDecoderInstance = NULL;
  jniInitialized = false;
}

// Public implementation, just hiding the internal implementing class
AacEldDecoder::AacEldDecoder(void *jvmHandle) : impl_(new AacEldDecImpl(jvmHandle)) {}

AacEldDecoder::~AacEldDecoder() {
  delete impl_;
}

bool AacEldDecoder::configure(std::vector<unsigned char>& asc) {
  return impl_->configure(asc);
}

bool AacEldDecoder::decode(std::vector<unsigned char>& inAU, std::vector<unsigned char>& outSamples) {
  return impl_->decode(inAU, outSamples);
}

void AacEldDecoder::close() {
  impl_->close();
}
