/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions
 *   for Use of Fraunhofer Example Source Code", which is provided as a
 *   separate document together with this file. The "Terms & Conditions for
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the
 *   Example Source Code.
 *
\***************************************************************************/

#include <jni.h>
#include <android/log.h>

#include <stdlib.h>

#include "JniEnvGuard.h"

#include "AacEldEncoder.h"

// Simple logging macro
#define LOG(...) __android_log_print(ANDROID_LOG_ERROR, "AacEldEncImpl", __VA_ARGS__)

// Internal encoder implementation
// hiding the JNI interaction from the rest
// of the native implementation
class AacEldEncoder::AacEldEncImpl {
public:
  // See comments in AacEldEncoder.h
  AacEldEncImpl(void *jvmHandle);
  ~AacEldEncImpl();

  bool configure(unsigned int sampleRate, unsigned int nChannels, unsigned int bitrate, std::vector<unsigned char>& asc);
  bool encode(std::vector<unsigned char>& inSamples, std::vector<unsigned char>& outAU);
  void close();

private:
  // Pointer to the JavaVM
  JavaVM*   javavm;
  // Reference to the encoder class
  jclass    aacEldEncoderClass;
  // Reference to the encoder instance
  jobject   aacEldEncoderInstance;
  // The methods of the encoder class: configure, encode and close
  jmethodID configureMethodId;
  jmethodID encodeMethodId;
  jmethodID closeMethodId;
  bool      jniInitialized;

  // Initializes the JNI members
  bool    initJni();
};

// Initializes the JNI specific member variables
bool AacEldEncoder::AacEldEncImpl::initJni() {
  // Always get a pointer to the JNIEnv first
  JniEnvGuard env(javavm);

  // Try to find the java class abstracting the encoder
  // In the demo code this is de.fraunhofer.iis.aac_eld_encdec.AacEldEncoder
  // This could also be in the same package as your other Java classes are but
  // it can also be separate
  jclass encoderClass = env->FindClass("de/fraunhofer/iis/aac_eld_encdec/AacEldEncoder");

  // If we have found the class we store a reference to it
  // because when multiple native threads are using the native codec implementation
  // the Android Java Class loader can mess things up
  // (see http://developer.android.com/training/articles/perf-jni.html#faq_FindClass)
  // and fail to find the class. When this function is only one time called from the main
  // thread, chances are good that the reference to the class stays valid for the rest of the
  // application lifetime
  if (encoderClass && !aacEldEncoderClass) // Store a global reference for this application
    aacEldEncoderClass = reinterpret_cast<jclass>(env->NewGlobalRef(encoderClass));


  if (!encoderClass) { // in case of an error, first check if
    if (aacEldEncoderClass) { // some thread got wild and we messed up the class loader stack
      encoderClass = aacEldEncoderClass; // try cached class if found before
      if(env->ExceptionCheck() == JNI_TRUE) { // and clear the pending exception that FindClass has already thrown
        env->ExceptionClear();
      }
    } else { // all bets are off - cannot find class
      LOG("ERROR in AacEldEncoder::AacEldEncImpl::initJni(): FindClass failed!");
      jthrowable exc = env->ExceptionOccurred();
      if (exc) {
        LOG("######## JVM Execption follows: ###########");
        env->ExceptionDescribe();
        env->ExceptionClear();
        LOG("###########################################");
      }
      return false;
    }
  }

  // Once the class is found, the method ids of that class can be obtained
  // (use javap -s to print the signatures of Java class methods)
  jmethodID encoder_ctor   = env->GetMethodID(encoderClass, "<init>", "()V");
  configureMethodId        = env->GetMethodID(encoderClass, "configure", "(III)[B");
  encodeMethodId           = env->GetMethodID(encoderClass, "encode", "([B)[B");
  closeMethodId            = env->GetMethodID(encoderClass, "close", "()V");

  // It is an error if one of these is not found
  if (!encoder_ctor || !configureMethodId || !encodeMethodId || !closeMethodId) {
    return false;
  }

  // If all methods are found, create a new instance of the AacEldEncoder object
  jobject encoderObj              = env->NewObject(encoderClass, encoder_ctor);

  if (!encoderObj) {
    return false;
  }

  // Finally create a new global reference (otherwise the
  // just created object will be garbage collected as soon
  // as the current JNI call returns to Java)
  aacEldEncoderInstance = env->NewGlobalRef(encoderObj);

  if (!aacEldEncoderInstance) {
    return false;
  }

  jniInitialized = true;

  return true;
}

AacEldEncoder::AacEldEncImpl::AacEldEncImpl(void *jvmHandle)
: javavm((JavaVM*)jvmHandle),
  aacEldEncoderClass(NULL),
  aacEldEncoderInstance(NULL),
  configureMethodId(NULL),
  encodeMethodId(NULL),
  closeMethodId(NULL),
  jniInitialized(false)
{}

AacEldEncoder::AacEldEncImpl::~AacEldEncImpl() {
  if (jniInitialized)
    close();
}

bool AacEldEncoder::AacEldEncImpl::configure(unsigned int sampleRate, unsigned int nChannels, unsigned int bitrate, std::vector<unsigned char>& asc) {
  if (!jniInitialized)
    if (!initJni())
      return false;

  JniEnvGuard env(javavm);

  jbyteArray resBuf = (jbyteArray) env->CallObjectMethod(aacEldEncoderInstance,
                                                          configureMethodId,
                                                          sampleRate,
                                                          nChannels,
                                                          bitrate);

   if (!resBuf) {
     return false;
   }

   jsize len  = env->GetArrayLength(resBuf);
   jbyte *buf = env->GetByteArrayElements(resBuf, 0);

   asc.clear();
   asc.resize(len);
   memcpy(&asc[0], buf, sizeof(unsigned char)*len);

   env->ReleaseByteArrayElements(resBuf, buf, 0);
   env->DeleteLocalRef(resBuf);

   return true;
}

bool AacEldEncoder::AacEldEncImpl::encode(std::vector<unsigned char>& inSamples, std::vector<unsigned char>& outAU) {

  JniEnvGuard env(javavm);

  jbyteArray inArray = env->NewByteArray(inSamples.size());
  jbyte *inBytes     = env->GetByteArrayElements(inArray, 0);

  memcpy(inBytes, &inSamples[0], sizeof(unsigned char)*inSamples.size());
  env->ReleaseByteArrayElements(inArray, inBytes, 0);

  jbyteArray resBuf = (jbyteArray) env->CallObjectMethod(aacEldEncoderInstance, encodeMethodId, inArray);
  env->DeleteLocalRef(inArray);

  if (!resBuf)
    return false;

  jsize resLen      = env->GetArrayLength(resBuf);
  jbyte *resByteBuf = env->GetByteArrayElements(resBuf, 0);
  outAU.clear();
  outAU.resize(resLen);
  memcpy(&outAU[0], resByteBuf, sizeof(unsigned char)*resLen);

  env->ReleaseByteArrayElements(resBuf, resByteBuf, 0);
  env->DeleteLocalRef(resBuf);

  return true;
}

void AacEldEncoder::AacEldEncImpl::close() {
  JniEnvGuard env(javavm);
  env->CallVoidMethod(aacEldEncoderInstance, closeMethodId);
  env->DeleteGlobalRef(aacEldEncoderInstance);
  aacEldEncoderInstance = NULL;
  jniInitialized = false;
}

// Public implementation, just hiding the internal implementing class
AacEldEncoder::AacEldEncoder(void *jvmHandle) : impl_(new AacEldEncImpl(jvmHandle)) {};

AacEldEncoder::~AacEldEncoder() {
  delete impl_;
}

bool AacEldEncoder::configure(unsigned int sampleRate, unsigned int nChannels, unsigned int bitrate, std::vector<unsigned char>& asc) {
  return impl_->configure(sampleRate, nChannels, bitrate, asc);
}

bool AacEldEncoder::encode(std::vector<unsigned char>& inSamples, std::vector<unsigned char>& outAU) {
  return impl_->encode(inSamples, outAU);
}

void AacEldEncoder::close() {
  impl_->close();
}
