/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions
 *   for Use of Fraunhofer Example Source Code", which is provided as a
 *   separate document together with this file. The "Terms & Conditions for
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the
 *   Example Source Code.
 *
\***************************************************************************/

#include "AudioProcessor.h"

#include <stdlib.h>

// No matter what your specific library needs for initialization, it absolutely
// needs a pointer to a JavaVM in order to
// access the Java MediaCodec instances later within the AacEldEncoder and AacEldDecoder classes.
// This handle is only required to initialize the encoder and decoder instances correctly
AudioProcessor::AudioProcessor(int sampleRate, int nChannels, int bitrate, int framesize, int noOfBytesPerSample, void *jvmHandle)
  : m_encoder(jvmHandle), // <- needs a handle to the JavaVM
    m_decoder(jvmHandle), // <- needs a handle to the JavaVM
    sampleRate(sampleRate),
    nChannels(nChannels),
    bitrate(bitrate),
    frameSize(framesize),
    noOfBytesPerSample(noOfBytesPerSample)
{
  // Initialize the AAC-ELD encoder with sample rate, bitrate, number of channels
  // and retrieve the audio specific config (ASC, see ISO/IEC 14496-3)
  std::vector<unsigned char> asc;
  m_encoder.configure(sampleRate, nChannels, bitrate, asc);

  // Use the ASC to initialize the AAC-ELD decoder
  m_decoder.configure(asc);
}

AudioProcessor::~AudioProcessor() {
  // Cleanup
  m_encoder.close();
  m_decoder.close();
}

void AudioProcessor::process(std::vector<unsigned char>& inSamples, std::vector<unsigned char>& outSamples) {

  // 1. Encode the input samples to an encoded access unit (AU)
  std::vector<unsigned char> encodedAU;
  m_encoder.encode(inSamples, encodedAU);

  // 2. Decode the just encoded AU to PCM samples again for playback
  outSamples.resize(frameSize*nChannels*noOfBytesPerSample);
  m_decoder.decode(encodedAU, outSamples);
}
