/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions
 *   for Use of Fraunhofer Example Source Code", which is provided as a
 *   separate document together with this file. The "Terms & Conditions for
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the
 *   Example Source Code.
 *
\***************************************************************************/

#include "JniEnvGuard.h"

#include <stdexcept>

JniEnvGuard::JniEnvGuard(JavaVM* vm, jint version) : vm_(vm), env_(NULL), threadAttached_(false) {
  // This method is used to obtain a JNIEnv pointer from a JavaVM instance.
  // The JNIEnv will become invalid when the method is not directly called
  // from a Java context (i.e. by calling a native Java method implementation),
  // but the JavaVM remains stable during the execution of a process
  // (however, on Android one cannot simply create a new JavaVM via JNI as
  // the function CreateJavaVM does not exists here, hence it is required
  // to pass the pointer around)
  jint jniResult = vm_->GetEnv(reinterpret_cast<void**>(&env_), version);

  if (jniResult == JNI_EDETACHED) { // Detached, attach
    jint rs = vm_->AttachCurrentThread(&env_, NULL);

    if (rs != JNI_OK) {
      throw std::runtime_error("Error attaching current thread to JNI VM");
    }

    threadAttached_ = true;
  } else if (jniResult != JNI_OK) {
    throw std::runtime_error("Error obtaining a reference to JNI environment");
  }

  if (env_ == NULL) {
    throw std::runtime_error("JNIEnv* invalid");
  }
}

JniEnvGuard::~JniEnvGuard() {
  if (threadAttached_) {
    vm_->DetachCurrentThread();
  }
}

JNIEnv* JniEnvGuard::operator->() {
  return env_;
}

JNIEnv const* JniEnvGuard::operator->() const {
  return env_;
}
