/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions
 *   for Use of Fraunhofer Example Source Code", which is provided as a
 *   separate document together with this file. The "Terms & Conditions for
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the
 *   Example Source Code.
 *
\***************************************************************************/

#include <jni.h>
#include <android/log.h>
#include <stdlib.h>

#include "AudioProcessor.h"

// Store field IDs of the native interface class
// where we will keep handles of the actual instances
// used for processing
static jfieldID AUDIO_PROCESSOR_FIELDID    = NULL;
static jfieldID AUDIO_INPUT_FRAME_FIELDID  = NULL;
static jfieldID AUDIO_OUTPUT_FRAME_FIELDID = NULL;

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Class:     de_fraunhofer_iis_android_aaceld_NativeAudioProcessor
 * Method:    init
 * Signature: (III)Z
 */
JNIEXPORT jboolean JNICALL Java_de_fraunhofer_iis_android_1aaceld_NativeAudioProcessor_init
  (JNIEnv *env, jobject obj, jint sampleRate, jint nChannels, jint bitrate, jint framesize, jint noOfBytesPerSample) {

  // Get field IDs that will store handles to the AudioProcessor, in and out frame classes
  jclass nativeAudioProcessorClass = env->GetObjectClass(obj);
  AUDIO_PROCESSOR_FIELDID          = env->GetFieldID(nativeAudioProcessorClass, "AudioProcessorHandle", "J");
  AUDIO_INPUT_FRAME_FIELDID        = env->GetFieldID(nativeAudioProcessorClass, "InputAudioFrameHandle", "J");
  AUDIO_OUTPUT_FRAME_FIELDID       = env->GetFieldID(nativeAudioProcessorClass, "OutputAudioFrameHandle", "J");

  JavaVM *vm = NULL;
  env->GetJavaVM(&vm);

  AudioProcessor *p = new AudioProcessor(sampleRate, nChannels, bitrate, framesize, noOfBytesPerSample, (void*)vm);

  const int audioFrameByteSize         = framesize * nChannels * noOfBytesPerSample;
  std::vector<unsigned char> *inFrame  = new std::vector<unsigned char>(audioFrameByteSize, 0);
  std::vector<unsigned char> *outFrame = new std::vector<unsigned char>(audioFrameByteSize, 0);

  env->SetLongField(obj, AUDIO_PROCESSOR_FIELDID, (jlong)p);
  env->SetLongField(obj, AUDIO_INPUT_FRAME_FIELDID, (jlong)inFrame);
  env->SetLongField(obj, AUDIO_OUTPUT_FRAME_FIELDID, (jlong)outFrame);
}

/*
 * Class:     de_fraunhofer_iis_android_aaceld_NativeAudioProcessor
 * Method:    process
 * Signature: ([B[B)Z
 */
JNIEXPORT jboolean JNICALL Java_de_fraunhofer_iis_android_1aaceld_NativeAudioProcessor_process
  (JNIEnv *env, jobject obj, jbyteArray inSamples, jbyteArray outSamples) {

  AudioProcessor *p    = (AudioProcessor*)env->GetLongField(obj, AUDIO_PROCESSOR_FIELDID);
  std::vector<unsigned char> *inFrame  = (std::vector<unsigned char>*)env->GetLongField(obj, AUDIO_INPUT_FRAME_FIELDID);
  std::vector<unsigned char> *outFrame = (std::vector<unsigned char>*)env->GetLongField(obj, AUDIO_OUTPUT_FRAME_FIELDID);

  jbyte *inBytes = env->GetByteArrayElements(inSamples, 0);
  memcpy(&(*inFrame)[0], inBytes, inFrame->size());
  env->ReleaseByteArrayElements(inSamples, inBytes, 0);

    p->process(*inFrame, *outFrame);

  jbyte *outBytes = env->GetByteArrayElements(outSamples, 0);
  memcpy(outBytes, &(*outFrame)[0], outFrame->size());
  env->ReleaseByteArrayElements(outSamples, outBytes, 0);

  return true;
}

JNIEXPORT jboolean JNICALL Java_de_fraunhofer_iis_android_1aaceld_NativeAudioProcessor_close
  (JNIEnv *env, jobject obj) {

  AudioProcessor *p    = (AudioProcessor*)env->GetLongField(obj, AUDIO_PROCESSOR_FIELDID);
  std::vector<unsigned char> *inFrame  = (std::vector<unsigned char>*)env->GetLongField(obj, AUDIO_INPUT_FRAME_FIELDID);
  std::vector<unsigned char> *outFrame = (std::vector<unsigned char>*)env->GetLongField(obj, AUDIO_OUTPUT_FRAME_FIELDID);
  delete p;
  delete inFrame;
  delete outFrame;
  env->SetLongField(obj, AUDIO_PROCESSOR_FIELDID, (jlong)0);
  env->SetLongField(obj, AUDIO_INPUT_FRAME_FIELDID, (jlong)0);
  env->SetLongField(obj, AUDIO_OUTPUT_FRAME_FIELDID, (jlong)0);

  return true;
}

#ifdef __cplusplus
}
#endif
