/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions 
 *   for Use of Fraunhofer Example Source Code", which is provided as a 
 *   separate document together with this file. The "Terms & Conditions for 
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the 
 *   Example Source Code.
 *
\***************************************************************************/

package de.fraunhofer.iis.aac_eld_encdec;

import java.nio.ByteBuffer;

import android.media.MediaCodec;
import android.media.MediaFormat;

public class AacEldDecoder {
  // The MediaCodec instance used as decoder
  private MediaCodec m_decoder;
  // Output buffer information
  private MediaCodec.BufferInfo m_bufferInfo;
  
  private Boolean m_isInitialized = false;
  
  // The constant mime-type and decoder name for AAC
  private final static String MIME_TYPE        = "audio/mp4a-latm";
  private final static String OMX_DECODER_NAME = "OMX.google.aac.decoder";
 
  public AacEldDecoder() {}
  
  public boolean configure(byte[] asc) {
    try {
      // Create a new MediaFormat using the Mime-Type, the samplerate and the channel count
      // Even though the sample rate and the channel count are also given by the ASC
      // we need to specify them here...
      MediaFormat mediaFormat = MediaFormat.createAudioFormat(MIME_TYPE, 0, 0);
     
      // This is obviously correct but very badly documented way
      // of how to pass the ASC to the decoder...
      ByteBuffer ascBuf       = ByteBuffer.wrap(asc);
      mediaFormat.setByteBuffer("csd-0", ascBuf); // "csd" probably stands for codec-specific-data
      
      // Create decoder instance using the decoder name
      m_decoder = MediaCodec.createByCodecName(OMX_DECODER_NAME);
      // Configure the decoder using the previously created MediaFormat instance
      m_decoder.configure(mediaFormat, null, null, 0);
    
      // Start the decoder
      m_decoder.start();
      
      // Create object for output buffer information
      m_bufferInfo = new MediaCodec.BufferInfo();
      m_isInitialized = true;

    } catch (Exception e) {
      System.out.println("ERROR configuring the decoder: " + e.getMessage());
      m_isInitialized = false;
    }

    return m_isInitialized;
  }
  
 public byte[] decode(byte[] au) {
    try {
      if (!m_isInitialized)
        return null;
      
      if (au != null) {
        // When we have a valid access unit (AU) we enqueue 
        // it as an input buffer to the decoder instance
        int decInBufIdx = m_decoder.dequeueInputBuffer(0);
        if (decInBufIdx >= 0) {
          m_decoder.getInputBuffers()[decInBufIdx].position(0);
          m_decoder.getInputBuffers()[decInBufIdx].put(au, 0, au.length);
          m_decoder.queueInputBuffer(decInBufIdx, 0, au.length, 0, 0);
        }
      }
    
      byte[] pcmbuf = null;
      int pcmbufPollCnt = 0;
      
      // Since the audio coding is implemented asynchronously internally
      // it is not guaranteed that upon enqueuing an AU the next call
      // to dequeueOutputBuffer will return the corresponding decoded PCM 
      // buffer or any output at all. In order to increase the probability 
      // that output data is generated, we poll the decoder a few times
      // (usually the output is generated after 40-60 polls - empirically
      // measured on a Galaxy Nexus running Android 4.2.2)
      while (pcmbuf == null && pcmbufPollCnt < 100) {
        int decBufIdx = m_decoder.dequeueOutputBuffer(m_bufferInfo, 0);
        if (decBufIdx >= 0) {
          pcmbuf = new byte[m_bufferInfo.size];
          m_decoder.getOutputBuffers()[decBufIdx].get(pcmbuf, 0, m_bufferInfo.size);
          m_decoder.getOutputBuffers()[decBufIdx].position(0);
          m_decoder.releaseOutputBuffer(decBufIdx, false);
        }
        ++pcmbufPollCnt;
      }
      
      return pcmbuf;
    } catch (Exception e) {
      return null;
    }
  }
  
  public void close() {
    // Clean up and release resources
    // WARNING: calling m_decoder.release()
    // will cause a segmentation fault(!) on
    // Samsung S III devices running Android 4.1!
    // The reason for this is still unknown.
    if (m_decoder != null) {
      m_isInitialized = false;
      m_decoder.flush();
      m_decoder.stop();
      m_decoder.release();
      m_decoder = null;
    }
  }
  
  @Override
  protected void finalize() throws Throwable {
      close();
      super.finalize();
  }
}
