/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions 
 *   for Use of Fraunhofer Example Source Code", which is provided as a 
 *   separate document together with this file. The "Terms & Conditions for 
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the 
 *   Example Source Code.
 *
\***************************************************************************/

package de.fraunhofer.iis.aac_eld_encdec;

import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaFormat;

public class AacEldEncoder {
  
  // The encoder instance
  private MediaCodec m_encoder;
  // Encoder output buffer information
  private MediaCodec.BufferInfo m_bufferInfo;
  // Initialization state
  private Boolean m_isInitialized = false;
  
  // MIME type and encoder name for AAC-ELD encoding
  private final static String MIME_TYPE        = "audio/mp4a-latm";
  private final static String OMX_ENCODER_NAME = "OMX.google.aac.encoder";
 
  public AacEldEncoder() {}

  public byte[] configure(int sampleRate, int channelCount, int bitrate) {
    try { // Handle any unexpected configuration issues here
      
      // Create and configure the media format for the encoder
      // This means, setting the sample rate, channel count and 
      // bitrate as well as the AAC-ELD profile level
      MediaFormat mediaFormat = MediaFormat.createAudioFormat(MIME_TYPE, sampleRate, channelCount);
      mediaFormat.setInteger(MediaFormat.KEY_AAC_PROFILE, MediaCodecInfo.CodecProfileLevel.AACObjectELD);
      mediaFormat.setInteger(MediaFormat.KEY_BIT_RATE, bitrate);
      
      // Create a codec instance and configure it to be an encoder
      m_encoder = MediaCodec.createByCodecName(OMX_ENCODER_NAME);
      m_encoder.configure(mediaFormat, null, null, MediaCodec.CONFIGURE_FLAG_ENCODE);
    
      // Start encoder
      m_encoder.start();
    
      //create object for output buffer information
      m_bufferInfo = new MediaCodec.BufferInfo();
   
      int ascPollCount = 0;
      byte[] aubuf = null;

      // See comment in encode() for explanation of polling
      while (aubuf == null && ascPollCount < 100) { 
        // Try to get the asc
        int encInBufIdx = -1;
        encInBufIdx = m_encoder.dequeueOutputBuffer(m_bufferInfo, 0);
        if (encInBufIdx >= 0) {
          if (m_bufferInfo.flags == MediaCodec.BUFFER_FLAG_CODEC_CONFIG) {
            aubuf = new byte[m_bufferInfo.size];
            m_encoder.getOutputBuffers()[encInBufIdx].get(aubuf, 0, m_bufferInfo.size);
            m_encoder.getOutputBuffers()[encInBufIdx].position(0);
            m_encoder.releaseOutputBuffer(encInBufIdx, false);
          }
        }
        ++ascPollCount;
      }
      
      if (aubuf != null) 
        m_isInitialized = true;
      
      return aubuf;
    } catch (Exception e) {
      System.out.println("ERROR configuring the encoder: " + e.getMessage());
      return null;
    }
  }
  
  public byte[] encode(byte[] pcmFrame) {
    try { 
      if (!m_isInitialized)
        return null;
    
      // When we have a valid PCM frame we enqueue 
      // it as an input buffer to the encoder instance
      if (pcmFrame != null) {
        int encInBufIdx = m_encoder.dequeueInputBuffer(0);
        if (encInBufIdx >= 0) {
          m_encoder.getInputBuffers()[encInBufIdx].position(0);
          m_encoder.getInputBuffers()[encInBufIdx].put(pcmFrame, 0, pcmFrame.length);
          m_encoder.queueInputBuffer(encInBufIdx, 0, pcmFrame.length, 0, 0);
        }
      }
    
      byte[] aubuf = null; 
      int aubufPollCnt = 0;
    
      // Since the audio coding is implemented asynchronously internally
      // it is not guaranteed that upon enqueuing a PCM buffer the next call
      // to dequeueOutputBuffer will return the corresponding encoded AU 
      // buffer or any output at all. In order to increase the probability 
      // that output data is generated, we poll the decoder a few times
      // (usually the output should be generated after 40-60 polls)
      while (aubuf == null && aubufPollCnt < 100) {
        int encInBufIdx = m_encoder.dequeueOutputBuffer(m_bufferInfo, 0);
        if (encInBufIdx >= 0) {
          aubuf = new byte[m_bufferInfo.size];
          m_encoder.getOutputBuffers()[encInBufIdx].get(aubuf, 0, m_bufferInfo.size);
          m_encoder.getOutputBuffers()[encInBufIdx].position(0);
          m_encoder.releaseOutputBuffer(encInBufIdx, false);
        }
        ++aubufPollCnt;
      }
    
      return aubuf;
    
    } catch (Exception e) { // Handle any unexpected encoding issues here
      return null;
    }
  }
  
  public void close() {
    // Clean up and release resources
    if (m_encoder != null) {
      m_isInitialized = false;
      m_encoder.flush();
      m_encoder.stop();
      m_encoder.release();
      m_encoder = null;
    }
  }
  
  @Override
  protected void finalize() throws Throwable {
      close();
      super.finalize();
  }
}
