/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions 
 *   for Use of Fraunhofer Example Source Code", which is provided as a 
 *   separate document together with this file. The "Terms & Conditions for 
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the 
 *   Example Source Code.
 *
\***************************************************************************/

package de.fraunhofer.iis.android_aaceld;

import android.app.Activity;
import android.media.AudioManager;
import android.os.Bundle;
import android.view.Menu;
import android.view.View;
import android.widget.Button;

import de.fraunhofer.iis.aac_eld_encdec.*;

// The main activity for the demo app
public class MainActivity extends Activity implements ISoundCardHandler, View.OnClickListener {

    // The simple sound card implementation
    private SoundCard            m_soundCard;
    
    // The Java interface to the encoder and the decoder
    private AacEldEncoder        m_encoder;
    private AacEldDecoder        m_decoder;
    
    // Select if we want to use the Java code path or the native code path
    private boolean              m_useNativeCodePath = false;
    
    // This is the interface to the native processing part of the application
    private NativeAudioProcessor m_audioProcessor;
    
    private boolean              m_running        = false;
    
    // For demo purposes, we use a fixed sampling rate, a mono channel configuration 
    // and an encoder bitrate of 65 kbps
    static private final int     encoderBitrate   = 65000;
    static private final int     frameSize        = 512;
    
    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        setVolumeControlStream(AudioManager.STREAM_VOICE_CALL);
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.main, menu);
        Button startStopButtonJava = (Button)findViewById(R.id.start_stop_java_btn);
        Button startStopButtonNative = (Button)findViewById(R.id.start_stop_native_btn);
        startStopButtonJava.setOnClickListener(this);
        startStopButtonNative.setOnClickListener(this);
        return true;
    }

    @Override
    public void onClick(View v) {
      
      if (v.getId() == R.id.start_stop_java_btn) {
        m_useNativeCodePath = false;
      } else if (v.getId() == R.id.start_stop_native_btn) {
        m_useNativeCodePath = true;
      }
      
      if (!m_running) {
        // Setup coding demo depending on user selection
        if (m_useNativeCodePath) {
          m_audioProcessor = new NativeAudioProcessor();
          m_audioProcessor.init(SoundCard.sampleRate, 
                                SoundCard.nChannels, 
                                MainActivity.encoderBitrate,
                                MainActivity.frameSize,
                                SoundCard.bytesPerSample);
        } else {
          m_encoder = new AacEldEncoder();
          m_decoder = new AacEldDecoder();
          m_decoder.configure(m_encoder.configure(SoundCard.sampleRate, SoundCard.nChannels, MainActivity.encoderBitrate));
        }
        
        m_soundCard = new SoundCard(this);
        m_soundCard.startCallback();
        m_running = true;
        
        if (m_useNativeCodePath) {
          Button startStopButtonNative = (Button)findViewById(R.id.start_stop_native_btn);
          startStopButtonNative.setText(R.string.stop_native_btn_str);
          Button startStopButtonJava = (Button)findViewById(R.id.start_stop_java_btn);
          startStopButtonJava.setEnabled(false);
        } else {
          Button startStopButtonJava = (Button)findViewById(R.id.start_stop_java_btn);
          startStopButtonJava.setText(R.string.stop_java_btn_str);
          Button startStopButtonNative = (Button)findViewById(R.id.start_stop_native_btn);
          startStopButtonNative.setEnabled(false);
        }
        
      } else { 
        m_soundCard.stopCallback();
        m_soundCard = null;
        m_running = false;
      
        if (m_useNativeCodePath) {
          m_audioProcessor.close();
          
          Button startStopButtonNative = (Button)findViewById(R.id.start_stop_native_btn);
          startStopButtonNative.setText(R.string.start_native_btn_str);
          Button startStopButtonJava = (Button)findViewById(R.id.start_stop_java_btn);
          startStopButtonJava.setEnabled(true);
        } else {
          m_encoder.close();
          m_decoder.close();
          
          Button startStopButtonJava = (Button)findViewById(R.id.start_stop_java_btn);
          startStopButtonJava.setText(R.string.start_java_btn_str);
          Button startStopButtonNative = (Button)findViewById(R.id.start_stop_native_btn);
          startStopButtonNative.setEnabled(true);
        }
      }
    }

    // Processes audio frames of 512 samples each
    private void processFrame(byte[] inFrame, byte[] outFrame) {
      
      if (m_useNativeCodePath) {
      
        // Performs encoding and decoding in native code
        m_audioProcessor.process(inFrame, outFrame);
      
      } else {
        // Use Java objects directly for encoding and decoding
        byte[] outAU  = m_encoder.encode(inFrame);
        byte[] tmpOut = m_decoder.decode(outAU);
        
        // Ensure that a valid sample buffer is returned even 
        // if decoding fails
        if (tmpOut == null) 
          java.util.Arrays.fill(outFrame, (byte) 0);
        else 
          System.arraycopy(tmpOut, 0, outFrame, 0, outFrame.length);
      }
      
    }
    
    @Override
    public boolean process(byte[] inputBuffer, byte[] outputBuffer) {
      
      // We need to split the input buffer into framesize-sized frames
      // because ELD requires frames containing 512 samples per packet
      if (inputBuffer.length != outputBuffer.length || inputBuffer.length % frameSize != 0) 
        throw new RuntimeException("Error! Cannot handle bogus sized audio buffers in this demo!");
      
      final int nChunks = inputBuffer.length / (frameSize*SoundCard.nChannels*SoundCard.bytesPerSample);
      
      byte[] inBuf = new byte[frameSize*SoundCard.nChannels*SoundCard.bytesPerSample];
      byte[] outBuf = new byte[frameSize*SoundCard.nChannels*SoundCard.bytesPerSample];
      
      for (int i = 0; i < nChunks; ++i) {
        System.arraycopy(inputBuffer,i*frameSize*SoundCard.nChannels*SoundCard.bytesPerSample, inBuf, 0, inBuf.length);
        
        processFrame(inBuf, outBuf);
        
        System.arraycopy(outBuf, 0, outputBuffer,i*frameSize*SoundCard.nChannels*SoundCard.bytesPerSample, outBuf.length);
      } 
      return true;
    }
}
