/***************************************************************************\
 *
 *               (C) copyright Fraunhofer - IIS (2013)
 *                        All Rights Reserved
 *
 *   Project:  Android AAC-ELD Example Source Code
 *
 *   By using this Example Source Code, you agree to the "Terms & Conditions 
 *   for Use of Fraunhofer Example Source Code", which is provided as a 
 *   separate document together with this file. The "Terms & Conditions for 
 *   Use of Fraunhofer Example Source Code" must be part of any redistribution
 *   of the Example Source Code or parts thereof, or modifications of the 
 *   Example Source Code.
 *
\***************************************************************************/

package de.fraunhofer.iis.android_aaceld;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioRecord;
import android.media.AudioTrack;
import android.media.MediaRecorder;

// Interface that has to be implemented by clients of the SoundCard class
interface ISoundCardHandler {
  public boolean process(byte[] inputBuffer, byte[] outputBuffer);
}

// Simple sound card abstraction using constant parameters for simplicity 
public class SoundCard {
  public static final int    sampleRate            = 44100;
  public static final int    nChannels             = 1;
  public static final int    bytesPerSample        = 2; // 16 bit PCM
  
  private int                 bufferSize           = 0;
  private ISoundCardHandler   soundCardHandler;
  private Recorder            recorder;
  private Player              player;


  // Method will return minimum audio buffer size for recording scaled up to the
  // nearest multiple of 512 
  static int recorderBufferSize() {
    int bufferSize = AudioRecord.getMinBufferSize(sampleRate, AudioFormat.CHANNEL_IN_MONO, AudioFormat.ENCODING_PCM_16BIT);
    // Framesize should be a multiple of 512
    int bytesPerFrame = 512 * nChannels * bytesPerSample;
    if (bufferSize % bytesPerFrame != 0)
      bufferSize = (bufferSize / bytesPerFrame + 1) * bytesPerFrame;
    return bufferSize;
  }

  // Method will return minimum audio buffer size for playback scaled up to the
  // nearest multiple of 512 
  static int playoutBufferSize() {
    int bufferSize = AudioTrack.getMinBufferSize(sampleRate, AudioFormat.CHANNEL_OUT_MONO, AudioFormat.ENCODING_PCM_16BIT);
    // Framesize should be a multiple of 512
    int bytesPerFrame = 512 * nChannels * bytesPerSample;
    if (bufferSize % bytesPerFrame != 0)
      bufferSize = (bufferSize / bytesPerFrame + 1) * bytesPerFrame;
    return bufferSize;
  }

  // Simple audio buffer holding bufferSize samples (minimum buffer size for playback & recording)
  private class AudioBuffer {
    byte[] data = new byte[bufferSize];
  }

  // Audio recorder running as a thread
  private class Recorder extends Thread {
    // New AudioRecord instance using fixed parameters 
    private final AudioRecord record  = new AudioRecord(MediaRecorder.AudioSource.VOICE_RECOGNITION,
                                                        sampleRate, AudioFormat.CHANNEL_IN_MONO, 
                                                        AudioFormat.ENCODING_PCM_16BIT, bufferSize);
    // Frames will hold successfully recorded audio frames (to be taken from the Player instance)
    private final ArrayBlockingQueue<AudioBuffer>  frames        = new ArrayBlockingQueue<AudioBuffer>(1);
    // FramesPool holds a pool of already allocated audio buffer memory (to avoid unnecessary allocations)
    private final LinkedBlockingQueue<AudioBuffer> framesPool    = new LinkedBlockingQueue<AudioBuffer>();

    public Recorder() {
      // create some audio buffers and pool them
      for (int i = 0; i != 3; ++i)
        framesPool.add(new AudioBuffer());
    }

    @Override
    // Run the recorder thread
    public void run() {
      android.os.Process.setThreadPriority(android.os.Process.THREAD_PRIORITY_URGENT_AUDIO);
      record.startRecording();

      while (!isInterrupted()) {
     // check if we have already an empty frame in the pool
        AudioBuffer frame = framesPool.poll(); 
        if (frame == null)
          frame = new AudioBuffer(); // if not, create one
        record.read(frame.data, 0, frame.data.length); // record samples

        try {
          // add recorded frame to the ArrayBlockingQueue
          if (!frames.offer(frame, 20, TimeUnit.MILLISECONDS)) {
            framesPool.add(frame); // if failed, drop samples
          }
        }
        catch (InterruptedException e) {
          interrupt();
        }
      }

      record.stop();
      record.release();
    }
  }

  private class Player extends Thread {
    // New AudioTrack instance using fixed parameters 
    private final AudioTrack player = new AudioTrack(AudioManager.STREAM_VOICE_CALL, sampleRate,
                                                     AudioFormat.CHANNEL_OUT_MONO, 
                                                     AudioFormat.ENCODING_PCM_16BIT,
                                                     bufferSize, AudioTrack.MODE_STREAM);
    // Buffer where output samples will be stored by the client
    private final byte[] outputBuffer = new byte[bufferSize];

    public Player() {}

    @Override
    public void run() {
      android.os.Process.setThreadPriority(android.os.Process.THREAD_PRIORITY_URGENT_AUDIO);
      player.play();
      
      while (!isInterrupted()) {
        AudioBuffer inputSamples;
        try {
          // Try to get recorded samples from the ArrayBlockingQueue
          inputSamples = recorder.frames.take();
        } catch (InterruptedException e) {
          interrupt();
          continue;
        }
        
        // Provide recorded samples to the client and receive output samples
        if (!soundCardHandler.process(inputSamples.data, outputBuffer))
          interrupt();
        // Free the input sample frame
        recorder.framesPool.add(inputSamples);
        
        // Playback audio
        player.write(outputBuffer, 0, outputBuffer.length);
   
      }

      player.stop();
      player.release();
    }
  }

  SoundCard(ISoundCardHandler soundCardHandler) {
    this.soundCardHandler = soundCardHandler;
    int recordBufferSize  = recorderBufferSize();
    int playoutBufferSize = playoutBufferSize();
    
    // Determine audio buffer sizes
    bufferSize            = Math.max(recordBufferSize, playoutBufferSize);
  }

  public void startCallback() {
    // Create and start recorder and player instances
    recorder = new Recorder();
    player   = new Player();
    recorder.start();
    player.start();
  }

  public void stopCallback() {
    // Stop and join recorder and player
    try {
      if (player != null)
        player.interrupt();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    try {
      if (recorder != null)
        recorder.interrupt();
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    // join player and recorder threads
    try {
      if (player != null)
        player.join();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    try {
      if (recorder != null)
        recorder.join();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    player   = null;
    recorder = null;
  }
}
