% /*************************************************************************************
%    Intel Corp.
%
%    Project Name:  60 GHz Enterprise Cubicle Channel Model
%    File Name:     cb_gen_inter_cls_fl.m
%    Authors:       A. Lomayev, R. Maslennikov
%    Version:       5.0
%    History:       May 2010 created
%
%  *************************************************************************************
%    Description:
% 
%    function returns NLOS space-temporal clusters parameters: azimuth/elevation angles
%    in [deg] relative to LOS direction for TX/RX, times of arrival in [ns] relative
%    to LOS time for Enterprise Cubicle environment for Far laptop location
%
%    [cls] = cb_gen_inter_cls_fl(N)
%
%    Inputs:
%
%       1. N - the number of realizations
%
%    Outputs:
%
%       Near laptop location:
%
%        1.  cls.toa   - times of arrival array, size 5xN
%        2.  cls.tx_az - TX azimuths array,      size 5xN
%        3.  cls.tx_el - TX elevations array,    size 5xN
%        4.  cls.rx_az - RX azimuths array,      size 5xN
%        5.  cls.rx_el - RX elevations array,    size 5xN
%
%    Row dimension in cls.x array:
%
%    1 - OW#1 cluster (cubicles #1,2)
%    2 - OW#2 cluster (cubicles #1,2)
%    3 - CW#1 cluster in cubicle #1 (or CW#3 cluster in cubicle #2)
%    4 - CW#2 cluster (cubicles #1,2)
%    5 - Table cluster (cubicles #1,2)
%
%  *************************************************************************************/
function [cls] = cb_gen_inter_cls_fl(N)

% select cubicle #1 or cubicle #2 with equal probabilities
p = rand(1,1);

if (p<0.5)
    [cls] = cubicle1(N);
else
    [cls] = cubicle2(N);
end

function [cls] = cubicle1(N)
    
% time of arrival
cls.toa(1,:) = 3 + 6.*rand(1,N);
cls.toa(2,:) = 6 + 7.*rand(1,N);
cls.toa(3,:) = ones(1,N);
cls.toa(4,:) = 7.*rand(1,N);
cls.toa(5,:) = ones(1,N);

% tx azimuth
cls.tx_az(1,:) = -49 + 29.*rand(1,N);
cls.tx_az(2,:) = 12.*rand(1,N);
cls.tx_az(3,:) = -15 + 13.*rand(1,N);
cls.tx_az(4,:) = -1 + 9.*rand(1,N);
cls.tx_az(5,:) = zeros(1,N);

% tx elevation
cls.tx_el(1,:) = -20 + 5.*rand(1,N);
cls.tx_el(2,:) = -17 + 4.*rand(1,N);
cls.tx_el(3,:) = -25 + 6.*rand(1,N);
cls.tx_el(4,:) = -22 + 5.*rand(1,N);
cls.tx_el(5,:) = -31 + 7.*rand(1,N);

% rx azimuth
cls.rx_az(1,:) = 50 + 18.*rand(1,N);
cls.rx_az(2,:) = -178 + 33.*rand(1,N);
cls.rx_az(3,:) = 39 + 11.*rand(1,N);
cls.rx_az(4,:) = -178 + 37.*rand(1,N);
cls.rx_az(5,:) = zeros(1,N);

% rx elevation
cls.rx_el(1,:) = -cls.tx_el(1,:);
cls.rx_el(2,:) = -cls.tx_el(2,:);
cls.rx_el(3,:) = -cls.tx_el(3,:);
cls.rx_el(4,:) = -cls.tx_el(4,:);
cls.rx_el(5,:) = cls.tx_el(5,:);

function [cls] = cubicle2(N)

% time of arrival
cls.toa(1,:) = 8 + 5.*rand(1,N);
cls.toa(2,:) = 6 + 7.*rand(1,N);
cls.toa(3,:) = ones(1,N);
cls.toa(4,:) = 7.*rand(1,N);
cls.toa(5,:) = ones(1,N);

% tx azimuth
cls.tx_az(1,:) = -85 + 34.*rand(1,N);
cls.tx_az(2,:) = -12.*rand(1,N);
cls.tx_az(3,:) = 15 - 13.*rand(1,N);
cls.tx_az(4,:) = 1 - 9.*rand(1,N);
cls.tx_az(5,:) = zeros(1,N);

% tx elevation
cls.tx_el(1,:) = -16 + 4.*rand(1,N);
cls.tx_el(2,:) = -17 + 4.*rand(1,N);
cls.tx_el(3,:) = -25 + 6.*rand(1,N);
cls.tx_el(4,:) = -22 + 5.*rand(1,N);
cls.tx_el(5,:) = -31 + 7.*rand(1,N);

% rx azimuth
cls.rx_az(1,:) = 37 + 14.*rand(1,N);
cls.rx_az(2,:) = 178 - 33.*rand(1,N);
cls.rx_az(3,:) = -39 - 11.*rand(1,N);
cls.rx_az(4,:) = 178 - 37.*rand(1,N);
cls.rx_az(5,:) = zeros(1,N);

% rx elevation
cls.rx_el(1,:) = -cls.tx_el(1,:);
cls.rx_el(2,:) = -cls.tx_el(2,:);
cls.rx_el(3,:) = -cls.tx_el(3,:);
cls.rx_el(4,:) = -cls.tx_el(4,:);
cls.rx_el(5,:) = cls.tx_el(5,:);