% /*************************************************************************************
%    Intel Corp.
%
%    Project Name:  60 GHz Enterprise Cubicle Channel Model
%    File Name:     cb_ref_loss.m
%    Authors:       A. Lomayev, R. Maslennikov
%    Version:       5.0
%    History:       May 2010 created
%
%  *************************************************************************************
%    Description:
% 
%    function returns clusters reflection loss according to normal distribution based
%    on the experimental data
%
%    [ref_loss] = cb_ref_loss()
%
%    Outputs:
%
%       1. ref_loss - reflection loss coefficients array
%
%    Inputs:
%
%       1. scen  - parameter selects laptop location scenario: 0 - near laptop location, 1 - far laptop location
%       2. ps    - polarization support parameter: 0 - TX/RX polarization vectors are not applied, 1 - polarization is applied
%       3. pol   - antennas polarization type 1x2 vector: pol(1) - polarization type for TX antenna, pol(2) - polarization type for RX antenna 
%
%
%    Row dimension in ref_loss array:
%
%    Near laptop location scenario
%
%    1 - OW#1 cluster
%    2 - CW#4 cluster
%    3 - Table cluster
%    4 - Table-Ceiling cluster
%
%    Far laptop location scenario
%
%    1 - OW#1 cluster (cubicles #1,2)
%    2 - OW#2 cluster (cubicles #1,2)
%    3 - CW#1 cluster in cubicle #1 (or CW#3 cluster in cubicle #2)
%    4 - CW#2 cluster (cubicles #1,2)
%    5 - Table cluster (cubicles #1,2)
%
%  *************************************************************************************/
function [ref_loss] = cb_ref_loss(scen,ps,pol)

switch (scen)
    case 0, % near laptop location
        switch (ps)
            case 0, % without polarization
                ref_loss_db = [ref_coef_db(0,1);ref_coef_db(1,3)];
                ref_loss = 10.^(ref_loss_db./20);
                
            case 1, % with polarization
                % polarization vectors for TX and RX antennas
                tx_pol = polarization(pol(1));
                rx_pol = polarization(pol(2));
                
                % OW#1 cluster                
                % reflection matrix
                R(1,1) = ref_coef(0,0);
                R(2,2) = ref_coef(0,1);
                R(1,2) = 0.4.*rand(1,1) - 0.2;
                R(2,1) = 0.4.*rand(1,1) - 0.2;
                ref_loss(1,:) = rx_pol'*R*tx_pol;
                
                % CW#4 cluster
                % reflection matrix
                R(1,1) = ref_coef(1,0);
                R(2,2) = ref_coef(1,0.8);
                R(1,2) = 0.4.*rand(1,1) - 0.2;
                R(2,1) = 0.4.*rand(1,1) - 0.2;
                ref_loss(2,:) = rx_pol'*R*tx_pol;
                
                % Table cluster                
                % reflection matrix
                R(1,1) = ref_coef(2,0.56);
                R(2,2) = ref_coef(2,0);
                R(1,2) = 0.1.*(2.*(randn(1,1)>0)-1);
                R(2,1) = 0.1.*(2.*(randn(1,1)>0)-1);
                ref_loss(3,:) = rx_pol'*R*tx_pol;
                
                % Table-ceiling cluster
                % reflection matrix
                R(1,1) = ref_coef(2,1);
                R(2,2) = ref_coef(2,0);                
                p12 = rand(1,1);
                if ((p12<0.25))
                    R(1,2) = -0.13;
                elseif ((p12>=0.25) & (p12<0.5))
                    R(1,2) = -0.06;
                elseif ((p12>=0.5) & (p12<0.75))
                    R(1,2) = 0.06;
                else
                    R(1,2) = 0.13;
                end
                p21 = rand(1,1);
                if ((p21<0.25))
                    R(2,1) = -0.13;
                elseif ((p21>=0.25) & (p21<0.5))
                    R(2,1) = -0.06;
                elseif ((p21>=0.5) & (p21<0.75))
                    R(2,1) = 0.06;
                else
                    R(2,1) = 0.13;
                end
                ref_loss(4,:) = rx_pol'*R*tx_pol;                
           otherwise,
           error('Prohibited value of "ps" parameter');
        end
       
    case 1, % far laptop location
        switch (ps)
            case 0, % without polarization
                ref_loss_db = [ref_coef_db(0,2);ref_coef_db(1,3)];
                ref_loss = 10.^(ref_loss_db./20);
                        
            case 1, % with polarization
                % polarization vectors for TX and RX antennas
                tx_pol = polarization(pol(1));
                rx_pol = polarization(pol(2));
                
                % OW#1 cluster
                % reflection matrix
                R(1,1) = ref_coef(0,0);
                R(2,2) = ref_coef(0,1);
                R(1,2) = 0.4.*rand(1,1) - 0.2;
                R(2,1) = 0.4.*rand(1,1) - 0.2;
                ref_loss(1,:) = rx_pol'*R*tx_pol;
                
                % OW#2 cluster
                % reflection matrix
                R(1,1) = ref_coef(0,0);
                R(2,2) = ref_coef(0,1);
                R(1,2) = 0.4.*rand(1,1) - 0.2;
                R(2,1) = 0.4.*rand(1,1) - 0.2;
                ref_loss(2,:) = rx_pol'*R*tx_pol;
                
                % CW#1 cluster
                % reflection matrix
                R(1,1) = ref_coef(1,0);
                R(2,2) = ref_coef(1,0.8);
                R(1,2) = 0.4.*rand(1,1) - 0.2;
                R(2,1) = 0.4.*rand(1,1) - 0.2;
                ref_loss(3,:) = rx_pol'*R*tx_pol;
                
                % CW#2 cluster
                % reflection matrix
                R(1,1) = ref_coef(1,0);
                R(2,2) = ref_coef(1,0.8);
                R(1,2) = 0.4.*rand(1,1) - 0.2;
                R(2,1) = 0.4.*rand(1,1) - 0.2;
                ref_loss(4,:) = rx_pol'*R*tx_pol;
                
                % Table cluster                
                % reflection matrix
                R(1,1) = ref_coef(2,0.56);
                R(2,2) = ref_coef(2,0);
                R(1,2) = 0.1.*(2.*(randn(1,1)>0)-1);
                R(2,1) = 0.1.*(2.*(randn(1,1)>0)-1);
                ref_loss(5,:) = rx_pol'*R*tx_pol;
           otherwise,
           error('Prohibited value of "ps" parameter');
        end
    otherwise,
        error('Prohibited value of "scen" parameter');
end


% generate reflection coefficient in dB
function y = ref_coef_db(type,size)

switch (type)
    case 0, % 1st order clusters reflected from OW
        mean_value = -10; % dB
        sigma   =   4; % dB
        tl = -2; % dB       
        y = mean_value + randn(size,1)*sigma;        
        idx = find(y > tl);        
        while ~isempty(idx)            
            y(idx) = mean_value + randn(length(idx),1)*sigma;
            idx = find(y > tl);
        end
        
    case 1,% 1st order clusters reflected from CW and table
           % 2nd order clusters reflected from table and then ceiling
        mean_value = -9; % dB
        sigma   =   1.2; % dB
        tl1 = -11; % dB
        tl2 = -5; % dB
        y = mean_value + randn(size,1)*sigma;  
        idx = find((y < tl1) | (y > tl2));
        while ~isempty(idx)           
            y(idx) = mean_value + randn(length(idx),1)*sigma;
            idx = find((y < tl1) | (y > tl2));
        end
end




% generate reflection coefficient
function y = ref_coef(type,p)

switch (type)
    case 0, % 1st order reflections from OW
        mean_value = -10; % db
        sigma = 4; % db
        tl = -2; % db
        y_db = mean_value + randn(1,1).*sigma;
        idx = find(y_db > tl);
        while ~isempty(idx)    
            y_db(idx) = mean_value + randn(1,1).*sigma;
            idx = find(y_db > tl);
        end
    case 1, % 1st order reflections from CW
        mean_value = -9; % db
        sigma = 1.7; % db
        tl1 = -11; % dB
        tl2 = -5; % dB
        y_db = mean_value + randn(1,1)*sigma;  
        idx = find((y_db < tl1) | (y_db > tl2));
        while ~isempty(idx)           
            y_db(idx) = mean_value + randn(1,1)*sigma;
            idx = find((y_db < tl1) | (y_db > tl2));
        end        
    case 2, % 1st order reflection from table
            % 2nd order reflection from table and then ceiling
        mean_value = -9; % db
        sigma = 1.2; % db
        tl1 = -11; % dB
        tl2 = -5; % dB
        y_db = mean_value + randn(1,1)*sigma;  
        idx = find((y_db < tl1) | (y_db > tl2));
        while ~isempty(idx)           
            y_db(idx) = mean_value + randn(1,1)*sigma;
            idx = find((y_db < tl1) | (y_db > tl2));
        end
end

y = 10.^(y_db./20);

prob = rand(1,1);

index = find(prob>p);

y(index) = -y(index);