% /*************************************************************************************
%    Intel Corp.
%
%    Project Name:  60 GHz Enterprise Cubicle Channel Model
%    File Name:     gen_cb_ch.m
%    Authors:       A. Lomayev, R. Maslennikov
%    Version:       5.0
%    History:       May 2010 created
%
%  *************************************************************************************
%    Description:
%
%    function returns space-temporal parameters for all 1st order
%    reflections for random receiver position in cubicle environment
%
%    [ch] = gen_cb_ch(cfg,ps,pol)
%
%
%    Inputs:
%
%       1. cfg.scen - parameter selects laptop location scenario: 0 - near laptop location, 1 - far laptop location
%       2. cfg.Plos - LOS (Line-of-Sight) parameter, permitted values: 0 - corresponds to NLOS scenario, 1 - corresponds to LOS scenario
%       3. cfg.D    - distance in [meters] between TX and RX
%       4. ps       - polarization support parameter: 0 - TX/RX polarization vectors are not applied, 1 - polarization is applied
%       5. pol      - antennas polarization 1x2 vector: pol(1) - polarization type for TX antenna, pol(2) - polarization type for RX antenna
%
%    Outputs:
%
%       1. ch.am    - amplitudes array
%       2. ch.toa   - times of arrival array
%       3. ch.tx_az - TX azimuths array
%       4. ch.tx_el - TX elevations array
%       5. ch.rx_az - RX azimuths array
%       6. ch.rx_el - RX elevations array
%
%  *************************************************************************************/
function [ch] = gen_cb_ch(cfg,ps,pol)

% distance between TX and RX
D = cfg.D;

% NLOS clusters probabilities
P = cb_cls_prob(cfg);

% generate NLOS clusters
switch (cfg.scen)
    case 0, % near laptop location
        cls = cb_gen_inter_cls_nl(1);
    case 1, % far laptop location
        cls = cb_gen_inter_cls_fl(1);
end

% clusters distances
dist = (cls.toa.*1e-9).*(3e8) + D;

% calculate attenuation constants for clusters
% attenuation due to propagation
lambda = 5e-3;
Gp = sqrt((lambda.^2)./((4.*pi.*(dist)).^2));

% attenuation due to reflection
Gr = cb_ref_loss(cfg.scen,ps,pol);

% attenuation due to penetration through CW
Gpn = cb_pen_loss(cfg.scen);

% attenuation constants
G = Gp.*Gr.*Gpn;

% generate intra-clusters structure
toa = [];
am = [];
tx_az = [];
tx_el = [];
rx_az = [];
rx_el = [];
        
% LOS cluster
if (cfg.Plos)
    toa = 0;
    am = sqrt((lambda.^2)./((4.*pi.*D).^2)).*exp(2j.*pi.*(D./lambda));
    G1 = am;
    
    if (ps)
        tx_pol = polarization(pol(1));
        rx_pol = polarization(pol(2));
        
        H = eye(2);
        H(1,2) = 0.1.*(2.*(randn(1,1)>0)-1);
        H(2,1) = 0.1.*(2.*(randn(1,1)>0)-1);
        
        pol_coef = rx_pol'*H*tx_pol;
        
        am = am.*pol_coef;
    end

    tx_az = 0;
    tx_el = -(asin(2./D).*180./pi);
    rx_az = 0;
    rx_el = -tx_el;
end

% NLOS clusters
switch (cfg.scen)
    case 0, % near laptop location
        for i=1:size(P,2) - 2
            if rand(1,1) < P(i)
                
                incls = cb_gen_intra_cls(cls.toa(i));
                toa = [toa; cls.toa(i) + incls.toa];
                am = [am; incls.am.*G(i)];
                tx_az = [tx_az; cls.tx_az(i) + incls.tx_az];
                tx_el = [tx_el; cls.tx_el(i) + incls.tx_el];
                rx_az = [rx_az; cls.rx_az(i) + incls.rx_az];
                rx_el = [rx_el; cls.rx_el(i) + incls.rx_el];            
            end
        end
        for i=size(P,2)-1:size(P,2)
            if rand(1,1) < P(i)
                toa = [toa; cls.toa(i)];                
                am = [am; G(i).*exp(2*pi*1i*rand(1,1))];
                tx_az = [tx_az; cls.tx_az(i)];
                tx_el = [tx_el; cls.tx_el(i)];
                rx_az = [rx_az; cls.rx_az(i)];
                rx_el = [rx_el; cls.rx_el(i)];
            end
        end
        
    case 1, % far laptop location
        for i=1:size(P,2) - 1
            if rand(1,1) < P(i)
                
                incls = cb_gen_intra_cls(cls.toa(i));
                toa = [toa; cls.toa(i) + incls.toa];
                am = [am; incls.am.*G(i)];
                tx_az = [tx_az; cls.tx_az(i) + incls.tx_az];
                tx_el = [tx_el; cls.tx_el(i) + incls.tx_el];
                rx_az = [rx_az; cls.rx_az(i) + incls.rx_az];
                rx_el = [rx_el; cls.rx_el(i) + incls.rx_el];            
            end
        end
        for i=size(P,2):size(P,2)
            if rand(1,1) < P(i)
                toa = [toa; cls.toa(i)];
                am = [am; G(i).*exp(2*pi*1i*rand(1,1))];
                tx_az = [tx_az; cls.tx_az(i)];
                tx_el = [tx_el; cls.tx_el(i)];
                rx_az = [rx_az; cls.rx_az(i)];
                rx_el = [rx_el; cls.rx_el(i)];
            end
        end
end

% check azimuth overflow
ind_tx_az = find(tx_az > 180);
tx_az(ind_tx_az) = tx_az(ind_tx_az) - 360;
ind_tx_az = find(tx_az < -180);
tx_az(ind_tx_az) = tx_az(ind_tx_az) + 360;

ind_rx_az = find(rx_az > 180);
rx_az(ind_rx_az) = rx_az(ind_rx_az) - 360;
ind_rx_az = find(rx_az < -180);
rx_az(ind_rx_az) = rx_az(ind_rx_az) + 360;

% check elevation overflow
ind_tx_el = find(tx_el > 90);
tx_el(ind_tx_el) = 180 - tx_el(ind_tx_el);
tx_az(ind_tx_el) = tx_az(ind_tx_el) + (-180).*(tx_az(ind_tx_el)>0) + 180.*(tx_az(ind_tx_el)<=0);

ind_tx_el = find(tx_el < -90);
tx_el(ind_tx_el) = -(180 + tx_el(ind_tx_el));
tx_az(ind_tx_el) = tx_az(ind_tx_el) + (-180).*(tx_az(ind_tx_el)>0) + 180.*(tx_az(ind_tx_el)<=0);

ind_rx_el = find(rx_el > 90);
rx_el(ind_rx_el) = 180 - rx_el(ind_rx_el);
rx_az(ind_rx_el) = rx_az(ind_rx_el) + (-180).*(rx_az(ind_rx_el)>0) + 180.*(rx_az(ind_rx_el)<=0);

ind_rx_el = find(rx_el < -90);
rx_el(ind_rx_el) = -(180 + rx_el(ind_rx_el));
rx_az(ind_rx_el) = rx_az(ind_rx_el) + (-180).*(rx_az(ind_rx_el)>0) + 180.*(rx_az(ind_rx_el)<=0);

% output channel structure
ch.am = am;
ch.toa = toa;
ch.tx_az = tx_az;
ch.tx_el = tx_el;
ch.rx_az = rx_az;
ch.rx_el = rx_el;