% /*************************************************************************************
%    Intel Corp.
%
%    Project Name:  60 GHz Living Room Channel Model
%    File Name:     gen_lr_ch.m
%    Authors:       A. Lomayev, R. Maslennikov
%    Version:       5.0
%    History:       May 2010 created
%
%  *************************************************************************************
%    Description:
%
%    function returns space-temporal parameters for all 1st and 2nd order
%    reflections for random transmitter position in LR environment
%
%    [ch] = gen_lr_ch(cfg,ps,pol)
%
%
%    Inputs:
%
%      1. cfg.D            - distance in [meters] between TX and RX
%      2. cfg.Plos         - LOS parameter: 0 - LOS between TX and RX is blocked, 1 - non-blocked
%      3. cfg.lr.P_1st_c   - probability that the cluster is present (i.e. not blocked) for the 1st order reflections from ceiling
%      4. cfg.lr.P_1st_f   - probability that the cluster is present (i.e. not blocked) for the 1st order reflections from floor
%      5. cfg.lr.P_1st_w   - probability that the cluster is present (i.e. not blocked) for the 1st order reflections from walls
%      6. cfg.lr.P_2nd_w   - probability that the cluster is present (i.e. not blocked) for the 2nd order reflections from two walls
%      7. cfg.lr.P_2nd_cf  - probability that the cluster is present (i.e. not blocked) for the 2nd order reflections from ceiling and floor
%      8. cfg.lr.P_2nd_wc  - probability that the cluster is present (i.e. not blocked) for the 2nd order wall-ceiling (ceiling-wall) reflections
%      9. cfg.lr.p_2nd_wf  - probability that the cluster is present (i.e. not blocked) for the 2nd order wall-floor (floor-wall) reflections
%      10. ps              - polarization support parameter: 0 - TX/RX polarization vectors are not applied, 1 - polarization is applied
%      11. pol             - antennas polarization 1x2 vector: pol(1) - polarization type for TX antenna, pol(2) - polarization type for RX antenna
%
%    Outputs:
%
%       1. ch.am    - amplitudes array
%       2. ch.toa   - times of arrival array
%       3. ch.tx_az - TX azimuths array
%       4. ch.tx_el - TX elevations array
%       5. ch.rx_az - RX azimuths array
%       6. ch.rx_el - RX elevations array
%
%  *************************************************************************************/
function [ch] = gen_lr_ch(cfg,ps,pol)

% distance between TX and RX
D = cfg.D;

% NLOS clusters probabilities
P = lr_cls_prob(cfg);

% generate NLOS clusters
cls = lr_gen_inter_cls(1);

% clusters distances
dist = (cls.toa.*1e-9).*(3e8) + D;

% calculate attenuation constants for clusters
% attenuation due to propagation
lambda = 5e-3;
Gp = sqrt((lambda.^2)./((4.*pi.*(dist)).^2));

% attenuation due to reflection
Gr = lr_ref_loss(ps,pol);

% attenuation constants
G = Gp.*Gr;

% generate intra-clusters structure
toa = [];
am = [];
tx_az = [];
tx_el = [];
rx_az = [];
rx_el = [];
        
% LOS cluster
if (cfg.Plos)
    toa = 0;
    am = sqrt((lambda.^2)./((4.*pi.*D).^2)).*exp(2j.*pi.*(D./lambda));
    
    if (ps)
        tx_pol = polarization(pol(1));
        rx_pol = polarization(pol(2));
        
        H = eye(2);
        H(1,2) = 0.1.*(2.*(randn(1,1)>0)-1);
        H(2,1) = 0.1.*(2.*(randn(1,1)>0)-1);
        
        pol_coef = rx_pol'*H*tx_pol;
        
        am = am.*pol_coef;
    end

    tx_az = 0;
    tx_el = 0;
    rx_az = 0;
    rx_el = 0;
end   
        
% NLOS clusters
while (isempty(toa) | (toa == 0))
    for i=1:18
        if rand(1,1) <= P(i)
            
            incls = lr_gen_intra_cls(cls.toa(i));
            toa = [toa; cls.toa(i) + incls.toa];
            am = [am; incls.am.*G(i)];
            tx_az = [tx_az; cls.tx_az(i) + incls.tx_az];
            tx_el = [tx_el; cls.tx_el(i) + incls.tx_el];
            rx_az = [rx_az; cls.rx_az(i) + incls.rx_az];
            rx_el = [rx_el; cls.rx_el(i) + incls.rx_el];            
        end
    end
end        
        

% check azimuth overflow
ind_tx_az = find(tx_az > 180);
tx_az(ind_tx_az) = tx_az(ind_tx_az) - 360;
ind_tx_az = find(tx_az < -180);
tx_az(ind_tx_az) = tx_az(ind_tx_az) + 360;

ind_rx_az = find(rx_az > 180);
rx_az(ind_rx_az) = rx_az(ind_rx_az) - 360;
ind_rx_az = find(rx_az < -180);
rx_az(ind_rx_az) = rx_az(ind_rx_az) + 360;

% check elevation overflow
ind_tx_el = find(tx_el > 90);
tx_el(ind_tx_el) = 180 - tx_el(ind_tx_el);
tx_az(ind_tx_el) = tx_az(ind_tx_el) + (-180).*(tx_az(ind_tx_el)>0) + 180.*(tx_az(ind_tx_el)<=0);

ind_tx_el = find(tx_el < -90);
tx_el(ind_tx_el) = -(180 + tx_el(ind_tx_el));
tx_az(ind_tx_el) = tx_az(ind_tx_el) + (-180).*(tx_az(ind_tx_el)>0) + 180.*(tx_az(ind_tx_el)<=0);

ind_rx_el = find(rx_el > 90);
rx_el(ind_rx_el) = 180 - rx_el(ind_rx_el);
rx_az(ind_rx_el) = rx_az(ind_rx_el) + (-180).*(rx_az(ind_rx_el)>0) + 180.*(rx_az(ind_rx_el)<=0);

ind_rx_el = find(rx_el < -90);
rx_el(ind_rx_el) = -(180 + rx_el(ind_rx_el));
rx_az(ind_rx_el) = rx_az(ind_rx_el) + (-180).*(rx_az(ind_rx_el)>0) + 180.*(rx_az(ind_rx_el)<=0);

% output channel structure
ch.am = am;
ch.toa = toa;
ch.tx_az = tx_az;
ch.tx_el = tx_el;
ch.rx_az = rx_az;
ch.rx_el = rx_el;