% /*************************************************************************************
%    Intel Corp.
%    (C) COPYRIGHT 2010
%
%    Project Name:  60 GHz Living Room Channel Model
%    File Name:     lr_ref_loss.m
%    Authors:       A. Lomayev, R. Maslennikov
%    Version:       5.0
%    History:       May 2010 created
%
%  *************************************************************************************
%    Description:
% 
%    function returns clusters reflection loss according to normal distribution based
%    on the experimental data
%
%    [ref_loss] = lr_ref_loss()
%
%    Outputs:
%
%       1. ref_loss - reflection loss coefficients array
%
%    Inputs:
%
%       1. ps    - polarization support parameter: 0 - TX/RX polarization vectors are not applied, 1 - polarization is applied
%       2. pol   - antennas polarization type 1x2 vector: pol(1) - polarization type for TX antenna, pol(2) - polarization type for RX antenna 
%
%
%    Row dimension in ref_loss array:
%
%    1,2                     - 1st order reflections from ceiling and floor
%    3,4,5                   - 1st order reflections from walls
%
%    6,7,8,9,10              - 2nd order reflections from two walls
%    11,12                   - 2nd order reflections from ceiling and floor
%    13,14,15,16,17,18       - 2nd order reflections from the wall and
%                              ceiling and from the wall and floor
%
%  *************************************************************************************/
function [ref_loss] = lr_ref_loss(ps,pol)

switch (ps)
    case 0, % without polarization        
        ref_loss_db = [ref_coef_db(0,5); ref_coef_db(1,13)];
                
        ref_loss = 10.^(ref_loss_db./20);
        
    case 1, % with polarization
        
        % polarization vectors for TX and RX antennas
        tx_pol = polarization(pol(1));
        rx_pol = polarization(pol(2));
        
        % 1st order reflections from ceiling and floor
        for i=1:2
            % reflection matrix
            R(1,1) = ref_coef(0,0.87);
            R(2,2) = ref_coef(0,0);
            R(1,2) = 0.1.*(2.*(randn(1,1)>0)-1);
            R(2,1) = 0.1.*(2.*(randn(1,1)>0)-1);
            
            ref_loss(i,:) = rx_pol'*R*tx_pol;
        end
        
        % 1st order reflections from walls
        for i=3:5
            % reflection matrix
            R(1,1) = ref_coef(0,0);
            R(2,2) = ref_coef(0,1);
            R(1,2) = 0.1.*(2.*(randn(1,1)>0)-1);
            R(2,1) = 0.1.*(2.*(randn(1,1)>0)-1);
            
            ref_loss(i,:) = rx_pol'*R*tx_pol;
        end
        
        % 2nd order reflections from two walls
        for i=6:10
            % reflection matrix
            R(1,1) = ref_coef(0,1);
            R(2,2) = ref_coef(0,0.87);
            R(1,2) = 0.2.*rand(1,1) - 0.1;
            R(2,1) = 0.2.*rand(1,1) - 0.1;
            
            ref_loss(i,:) = rx_pol'*R*tx_pol;
        end
        
        % 2nd order reflections from ceiling and floor
        for i=11:12
            % reflection matrix
            R(1,1) = ref_coef(0,1);
            R(2,2) = ref_coef(0,1);
            R(1,2) = 0.2.*rand(1,1) - 0.1;
            R(2,1) = 0.2.*rand(1,1) - 0.1;
            
            ref_loss(i,:) = rx_pol'*R*tx_pol;
        end
        
        % 2nd order reflections from the wall and ceiling and from the wall
        % and floor
        for i=13:18
            % reflection matrix
            R(1,1) = ref_coef(0,0.93);
            R(2,2) = ref_coef(0,0);
            R(1,2) = 0.3.*rand(1,1) - 0.15;
            R(2,1) = 0.3.*rand(1,1) - 0.15;
            
            ref_loss(i,:) = rx_pol'*R*tx_pol;
        end
        
    otherwise,
        error('Prohibited value of "ps" parameter');
end

% generate reflection coefficient in dB
function y = ref_coef_db(type,size)

switch (type)
    case 0, % 1st order
        mean_value = -10; % dB
        sigma   =   4; % dB
    case 1,% 2nd order
        mean_value = -16; % dB
        sigma   =   5; % dB
end

y = mean_value + randn(size,1)*sigma;

idx = find(y > -2);

while ~isempty(idx)
    
    y(idx) = mean_value + randn(length(idx),1)*sigma;
    idx = find(y > -2);
end


% generate reflection coefficient
function y = ref_coef(type,p)

switch (type)
    case 0, % 1st order reflections
        mean_value = -10; % db
        sigma = 4; % db
    case 1, % 2nd order reflections
        mean_value = -16; % db
        sigma = 5; % db        
end

y_db = mean_value + randn(1,1).*sigma;

idx = find(y_db > -2);

while ~isempty(idx)
    
    y_db(idx) = mean_value + randn(1,1).*sigma;
    idx = find(y_db > -2);
end

y = 10.^(y_db./20);

prob = rand(1,1);

index = find(prob>p);

y(index) = -y(index);