% This source code is a part of Master Thesis titled
% "Resource Allocation in a Distributed Antenna System"
% at
%
% Institute for Digital Communications
% Friedrich-Alexander University Erlangen-Nürnberg (FAU)
% Wetterkreuz 15, D-91058 Erlangen, Germany
% Date: 9th January 2017
%
% Author: Meysam Goodarzi (meysam.goodarzi@fau.de)
% Second contact: Aravindh Krishnamoorthy (aravindh.krishnamoorthy@fau.de)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% This function has been developed based on Quadriga Radio Channel Model.
% Detail explanation can be found in "Quadriga --> 
% quadriga_documentation_v1.4.1-551 [Section 2, Software Structure]"

    function hcoeff = channel_O2I( BandWidth,no_subcarriers,CenterFrequency,NoTx,BS_antenna_model,no_TxArrayElements,no_URUs,room_side,BS_to_S_dist,GeneralScenario )
    
    % URUs are distributed uniformly over ceiling
    grid = ceil(sqrt(no_URUs));
    dist = room_side/(grid -1);
    xGrid = repmat([-room_side/2:dist:room_side/2],1,grid) ;
    yGrid = repmat([-room_side/2:dist:room_side/2],grid,1) ;
    yGrid = reshape(yGrid,1,[]);
    
    RxPositionsX = xGrid(1:no_URUs) + BS_to_S_dist ;
    RxPositionsY = yGrid(1:no_URUs);
     
    % Height of the room
    RxPositionsZ = [3*ones(1,no_URUs)] ;
    
    
    TxPositionsX = [zeros(1,NoTx)] ;    
    TxPositionsY = [zeros(1,NoTx)] ;
    TxPositionsZ = [25*ones(1,NoTx)] ;

    %Antenna array on base station (vector below will be added to the
    %previously defined vector TxPositionsZ
    ArrayPosition =[-2.66 -2.28 -1.90 -1.52 -1.14 -0.76 -0.38 0];

    % simulation parameters
    sim = simulation_parameters;
    sim.center_frequency = CenterFrequency ;
    sim.sample_density = 4 ;                  % Number of samples per half_wave length
  % sim.use_absolute_delays = 1 ;             % '0' normalized to LOS, '1' absolute value
  % sim.drifting_precision = 0;               % please refer to Quadriga class 'simulations_parameters'
    sim.map_resolution = 10;                  % correlation map resolution
    
    % Layout setting
    l = layout(sim) ;
    l.no_tx = NoTx ;
    l.no_rx = no_URUs ;
    
    % RXs and TXs positions
    l.tx_position(1,:) = TxPositionsX ;
    l.tx_position(2,:) = TxPositionsY ;
    l.tx_position(3,:) = TxPositionsZ ;
    l.rx_position(1,:) = RxPositionsX ;
    l.rx_position(2,:) = RxPositionsY ;
    l.rx_position(3,:) = RxPositionsZ ;


    % Antenna specifications
    if BS_antenna_model == '3gpp'
        % each antenna element type is "3gpp-macro"
        for i=1:NoTx
            txAnt = array('3gpp-macro',60,10,40);
            txAnt.generate('3gpp-macro',[],60,10,40);
            txAnt.no_elements = no_TxArrayElements ;
            txAnt.element_position(1,:) = [zeros(1,no_TxArrayElements)];
            txAnt.element_position(2,:) = [zeros(1,no_TxArrayElements)];
            txAnt.element_position(3,:) = ArrayPosition(i,:) ;
            l.tx_array(i)= txAnt;
        end
        
        % Array of 8 omni antenna with 10cm distance
    elseif BS_antenna_model == 'ula8'
        txAnt = array('ula8');
        l.tx_array= txAnt;
        
        % Isotropic antenna 
    elseif BS_antenna_model == 'omni'
        txAnt = array('omni');
        l.tx_array= txAnt;
    end
    
    
    % If the BS antenna needs to be rotated
    % l.tx_array(1,1).rotate_pattern(-10,'y')
    
    % Antenna type of receiver (URUs)
     l.rx_array.generate('omni');

    % l.tx_array.calc_gain();    % Antenna gains can be seen by this command

    % Antennas can be Visualized by this command
    %l.tx_array.visualize;

    % Scenario specification
    l.track.set_scenario(GeneralScenario);
    
    % Scenario can be Visualized by this command
    % l.visualize();

    % Generating LSP Parameters
    p = l.create_parameter_sets();
    p.update_parameters();

    % Generating channel coefficients
    c = p.get_channels;
    
    hf = [];
    % outer for-loop is to generate channel coefficient for each recieve
    % antenna
    for j=1:l.no_rx
        % Inner for loop is for the case when we have more than one BS
        for i=1:l.no_tx
        h = c(j+(i-1)*l.no_rx).fr(BandWidth,no_subcarriers,1);
        temp = squeeze(h) ;
        hf(j,(i-1)*l.tx_array(1,i).no_elements+1:(i)*l.tx_array(1,i).no_elements,:) = temp(:,:);
        end
    end
    
    % channel coefficients
    hcoeff = hf ;
    end

