% This source code is a part of Master Thesis titled
% "Resource Allocation in a Distributed Antenna System"
% at
%
% Institute for Digital Communications
% Friedrich-Alexander University Erlangen-Nürnberg (FAU)
% Wetterkreuz 15, D-91058 Erlangen, Germany
% Date: 9th January 2017
%
% Author: Meysam Goodarzi (meysam.goodarzi@fau.de)
% Second contact: Aravindh Krishnamoorthy (aravindh.krishnamoorthy@fau.de)
%


% This function is developed based on following Master thesis
% "Resource Allocation in a Distributed Antenna System"
% All the equation numbers that we mention in this code, are refered to
% the same equation numbers in above thesis.

% format "Eq.ChapterNumber.EquationNumber" is used to refer to an equation, ex: Eq.4.96 refers to
% equation 96 in chapter 4 in the thesis.


    function [Y_P_BS_S_opt , Y_P_S_UE_opt , Y_P_UE_S_opt , Y_P_S_BS_opt , Y_s_DL , sumRate] = opt_ind(h_BS_S,h_S_UE,pt_max...
        ,pt_URU_max_DLorUL,no_subcarriers,no_URUs,no_UE,pt_UEk_max)

    % max BS power Watt per subcarrier
    pt_max_BS_lin = (10^(pt_max/10))*1e-3  ;

    % max of per URU per subcarrier Power  Watt
    pt_URU_max_DLorUL_lin = 10^(pt_URU_max_DLorUL/10) *1e-3  ;

    % max of UE power Watt
    pt_UEk_max_lin = 10^(pt_UEk_max/10)*1e-3   ;

    % Noise power calculation
    temperature = 300;
    Boltzmanns_constant = 1.38*10^-23;
    noise_figure_dB = 8;
    noise_figure_linear = 10^(noise_figure_dB./10);
    subcarrier_bw = 15e+3  ;                             %KHz
    thermal_noise = subcarrier_bw*temperature*Boltzmanns_constant*noise_figure_linear ;  % Watt

    % Calculating channel to noise ratio (CNR)
    h_BS_S = h_BS_S./sqrt(thermal_noise) ;
    h_S_UE = h_S_UE./sqrt(thermal_noise) ;
    gamma_S_UE = abs(h_S_UE).^2;


    %% Algorithm implementation

    no_iterations = 50;
    
    % Downlink matrices initialization
    P_BS_S_opt=[];
    % random initialization
    % xx = rand(1,no_subcarriers); 
    % P_BS_S_opt(:,:,1) = [pt_max_BS_lin/no_subcarriers/8.*xx/sum(xx(:));zeros(7,no_subcarriers)];
    
    % equal power distribution
    % P_BS_S_opt(:,:,1) = pt_max_BS_lin/no_subcarriers/8.*ones(8,no_subcarriers);   
    
    % giving all the power to first antenna while distributing equally
    % among all subcarriers
    %P_BS_S_opt(:,:,:,1) = pt_max_BS_lin/no_UE/no_subcarriers.*[ones(no_UE,1,no_subcarriers) zeros(no_UE,7,no_subcarriers)];
    
    P_S_UE_opt(:,:,:,1) = zeros(no_UE,no_URUs,no_subcarriers);

    % SVD decomposition for each subcarrier's channel
    sig_BS_S = [];
    for c = 1:no_subcarriers
        [U_tot(:,:,c),S(:,:,c),V_tot(:,:,c)] = svd(h_BS_S(:,:,c)) ;
        sig_BS_S(:,c) = diag(S(:,:,c)'*S(:,:,c));
    end
    % outdoor-to-indoor channel CNR for each subcarrier, \Lambda_B_S in Eq.28
    gamma_BS_S = sig_BS_S ;
    N_S = size(gamma_BS_S,1);
    % addition of a new dimension for multiple user case
    gamma_BS_S = permute(repmat(gamma_BS_S,1,1,no_UE),[3 1 2]);
    
    
    s_DL_BS_S = 1./no_UE.*ones(no_UE,N_S,no_subcarriers);                         % DL transmission time portion
    s_DL_S_UE = 1./no_UE.*ones(no_UE,no_URUs,no_subcarriers);                         % DL transmission time portion
    
        
    % identity matrices 
    Ib = eye(size(sig_BS_S,1)) ;
    Is = eye(no_URUs);
                                   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        % Please note that the problem with assumption of non-joint
        % processing between URUs has been solved only for downlink
        % therefore in the simulation package only the implementation for
        % downlink scenario can be found.
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    

    
    for h = 2:no_iterations

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%%%%%%%%%%%%%%%%%%%%%%      DOWNLINK     %%%%%%%%%%%%%%%%%%%%%%
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
                % Calculating g_B_S in Eq.4.13
        G=[];
        g=[];
        for k = 1:no_UE
        for c = 1:no_subcarriers
            G(:,:) = diag(P_S_UE_opt(k,:,c,h-1)).^0.5*h_BS_S(:,:,c)*V_tot(:,:,c);
            g(k,:,c) = (sum(G.*conj(G))).';
        end
        end
        % Updating Langrangian Multiplier for Maximum BS DL power, Eq.4.13 (using BiSection
        % Method)
        
        % Lambda and Delta are calculated alternatively
        
            Delta = [];
            Lambda = [];
            Delta(1) = 0 ;
        
        for l = 2:50
                 
            Lambda0 = 1e-15;
            stepsize = 1e-4 ;
            counter = 0 ;
            temp_grad_dec = 0;
            P_BS_S_opt_temp = [] ;
            while counter <1e+15
                % Eq.4.13
                P_BS_S_opt_temp  = s_DL_BS_S.*max(0,1./(Lambda0+Delta(l-1)*g) - 1./gamma_BS_S) ;

                % C1 in Eq.4.9
                tol =  sum(P_BS_S_opt_temp(:)) - pt_max_BS_lin ;                
                % Checking whether the C1 is met (Bisection method)
                if counter==0 && tol <0
                    Lambda(l)=0;
                    break
                end
                if tol < temp_grad_dec & tol < 0
                    if abs(tol) > pt_max_BS_lin/1000
                        Lambda0 = Lambda0 - stepsize ;
                        stepsize = stepsize/2;
                        tol = temp_grad_dec;
                    else

                        P_BS_S_opt(:,:,:,h)= P_BS_S_opt_temp;
                        %disp('Constraint P_BS_S_opt is met');
                        break
                    end
                else
                    stepsize=2*stepsize;
                end
                temp_grad_dec = tol ;
                Lambda0 = Lambda0 + stepsize ;
                counter = counter + 1;
            end

            Lambda(1,l) = Lambda0;
            
            Delta0 = 1e-15 ;
            stepsize = 1e-4 ;
            counter = 0 ;
            temp_grad_dec = 0;
            P_BS_S_opt_temp = [] ;
            while counter <1e+15
                % Eq.4.13
                P_BS_S_opt_temp  = s_DL_BS_S.*max(0,1./(Lambda(l)+Delta0*g) - 1./gamma_BS_S) ;
                             
                % C2 in Eq.4.9                
                tol =  sum(sum(sum(P_BS_S_opt_temp.*g))) + sum(sum(sum(P_S_UE_opt(:,:,:,h-1)))) - pt_URU_max_DLorUL_lin;
                
                % Checking whether the C2 is met (Bisection method)
                if counter == 0 && tol <0
                    Delta0 = 0;
                    break
                end
                if tol < temp_grad_dec & tol < 0
                    if abs(tol) >  pt_URU_max_DLorUL_lin/1000
                        Delta0 = Delta0 - stepsize ;
                        stepsize = stepsize/2;
                        tol = temp_grad_dec;
                    else
                        P_BS_S_opt(:,:,:,h)= P_BS_S_opt_temp;
                        break
                    end
                else
                    stepsize=2*stepsize;
                end
                temp_grad_dec = tol ;
                Delta0 = Delta0 + stepsize ;
                counter = counter + 1;
            end
            Delta(1,l)= Delta0;
            
            % Checking the convergence of lagrangian multipliers in Eq.4.13
            logic1=[];
            logic2=[];
            logic1 = abs(Lambda(1,l)-Lambda(1,l-1)) <= 1e-4 ;
            logic2 = abs(Delta(1,l)-Delta(1,l-1)) <= 1e-4 ;
            if logic1 ==1 && logic2 ==1
                break
            end
        end
            
            
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   %
   %                       Calculating URUs powers
   %
   %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        % Calculating g_S_UE in Eq.4.14
        B = [];
        g = [];
        for k = 1:no_UE
        for c = 1:no_subcarriers
            B(:,:) = h_BS_S(:,:,c)*V_tot(:,:,c)* diag(P_BS_S_opt(k,:,c,h)).^0.5;
            g(k,:,c) = diag(B*B');
        end
        end

    % Updating Langrangian Multiplier for Maximum URUs DL power, Eq.4.14 (using BiSection
    % Method)

        Delta = 1e-15;                          % Lagrangian multiplier (initialization)
        counter = 0;                            % introducing a counter to limit the iteration in which we look for right Lagrangian multiplier
        stepsize = 1 ;                          % step size for finding the proper lagrangian multiplier
        temp_grad_dec = 0;
        P_S_UE_opt_temp = [] ;
        while counter <1e+15
            % Eq.4.14         
            P_S_UE_opt_temp =  s_DL_S_UE.*max(0,( sqrt(4./(gamma_S_UE.*Delta.*(g+1)) + (1./gamma_S_UE).^2) - 1./gamma_S_UE )./2 ) ;
            % C2 in Eq.4.9
            tol = sum(sum(sum(P_S_UE_opt_temp.*(g+1)))) -  pt_URU_max_DLorUL_lin;
            
            % Checking whether the C2 is met (Bisection method)
            if tol < temp_grad_dec & tol < 0
                if abs(tol) > pt_URU_max_DLorUL_lin/1000
                    Delta = Delta - stepsize ;
                    stepsize = stepsize/2;
                    tol = temp_grad_dec;
                else
                    P_S_UE_opt(:,:,:,h) = P_S_UE_opt_temp;
                    
                    break
                end
            else
                stepsize=2*stepsize;
            end
            temp_grad_dec = tol ;
            Delta = Delta + stepsize ;
            counter = counter + 1;
        end
        if counter == 1e+15
            display('Constraint P_S_UE_opt CONNOT be met')
        end

        % This part is to be activated if you wish to initialize Relays
        % instead of BS. It's highly recommended to initialize using either
        % the following one or allocating all the power to one antenna at
        % BS while distributing it equally among the subcarriers

%                if h==2
%         
%                   P_S_UE_opt(:,:,:,h) = zeros(no_UE,no_URUs,no_subcarriers);
%               end


        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
        temp = [];
        SNR_BS_S = P_BS_S_opt(:,:,:,h).* gamma_BS_S;
        SNR_S_UE = P_S_UE_opt(:,:,:,h).* gamma_S_UE;
        
        % here one of the criterias can be chosen
        % temp = sum((log2(1+SNR_BS_S) - SNR_BS_S./(1+SNR_BS_S)),2) + sum((log2(SNR_S_UE./(SNR_S_UE+1)) - 1./(1+SNR_S_UE) - Delta.*g.*P_S_UE_opt(:,:,:,h)) ,2);
         temp = sum((log2(1+SNR_BS_S)- Lambda(1,l).*P_BS_S_opt(:,:,:,h)),2) + sum((log2(SNR_S_UE./(SNR_S_UE+1)) - Delta.*(g+1).*P_S_UE_opt(:,:,:,h)) ,2);
        M_DL = sum(temp,2);
        temp = max(M_DL,[],1);
        s_DL_BS_S = zeros(no_UE,N_S,no_subcarriers);
        s_DL_S_UE = zeros(no_UE,no_URUs,no_subcarriers);
        for k = 1:no_UE
            logic0 = (M_DL(k,:,:)==temp)+1e-6;
            s_DL_BS_S(k,:,:) = repmat(logic0,N_S,1);
            s_DL_S_UE(k,:,:) = repmat(logic0,no_URUs,1);
        end
        % Checking the convergence of BS and URUs powers
        logic1=[];
        logic2=[];
        logic1 = (abs(P_BS_S_opt(:,:,:,h)-P_BS_S_opt(:,:,:,h-1))<= 1e-5);
        logic2 = (abs(P_S_UE_opt(:,:,:,h)-P_S_UE_opt(:,:,:,h-1))<= 1e-6);

        if all(logic1(:)==1) && all(logic2(:)==1)

            break
        end
    end

    % Returning the outputs
    Y_P_BS_S_opt = P_BS_S_opt(:,:,:,h);
    Y_P_S_UE_opt = P_S_UE_opt(:,:,:,h);
    
    % if the code is improved such that Uplink is considered then this part
    % is to be activated
    Y_P_UE_S_opt = 0;
    Y_P_S_BS_opt = 0;
    
    % Since we have just DL, we dedicate the whole time portion to downlink
    Y_s_DL = s_DL_BS_S ;

    %Cheking constraints
%      pt_max_BS_lin/1000
%      sum(Y_P_BS_S_opt(:)) - pt_max_BS_lin
%      pt_URU_max_DLorUL_lin/1000
%      sum(sum(sum(P_S_UE_opt_temp.*(g+1)))) -  pt_URU_max_DLorUL_lin
   
    % sumRate Calculation
    sumRate = 0;
    Rate_DL =0;
    for k = 1:no_UE
    for c = 1:no_subcarriers
        A = [];
        B = [];
        A = diag(h_S_UE(k,:,c))*diag(Y_P_S_UE_opt(k,:,c).*s_DL_S_UE(k,:,c)).^0.5*h_BS_S(:,:,c)*V_tot(:,:,c)*diag(Y_P_BS_S_opt(k,:,c).*s_DL_BS_S(k,:,c)).^0.5 ;
        B = inv(diag(gamma_S_UE(k,:,c))*diag(Y_P_S_UE_opt(k,:,c).*s_DL_BS_S(k,:,c)) + diag(s_DL_S_UE(k,:,c)).*Is);
        Rate_DL = s_DL_BS_S(k,1,c)*subcarrier_bw*log2(det( Ib + A'*B*A)) ;

        sumRate = sumRate + real(Rate_DL) ;
    end
    end
 
    end










