% This source code is a part of Master Thesis titled
% "Resource Allocation in a Distributed Antenna System"
% at
%
% Institute for Digital Communications
% Friedrich-Alexander University Erlangen-Nürnberg (FAU)
% Wetterkreuz 15, D-91058 Erlangen, Germany
% Date: 9th January 2017
%
% Author: Meysam Goodarzi (meysam.goodarzi@fau.de)
% Second contact: Aravindh Krishnamoorthy (aravindh.krishnamoorthy@fau.de)
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% This function is developed based on following Journal paper
% "Energy-Efficient 5G Outdoor-to-Indoor Communication: SUDAS Over Licensed and Unlicensed Spectrum"
% All the equation numbers that we mention in this code, are refered to
% the same equation numbers in above paper.

% format "Eq.number" is used to refer to an equation, ex: Eq.91 refers to
% equation 91.

function [Y_P_BS_S_opt , Y_P_S_UE_opt , Y_P_UE_S_opt , Y_P_S_BS_opt , Y_s_DL , sumRate] = opt_joint(h_BS_S,h_S_UE,pt_max...
    ,pt_URU_max_DLorUL,no_subcarriers,no_URUs,no_UE,pt_UEk_max)

% max BS power (Watt)
pt_max_BS_lin = (10^(pt_max/10))*1e-3   ;

% max of per URUs Power (Watt)
pt_URU_max_DLorUL_lin = 10^(pt_URU_max_DLorUL/10) *1e-3   ;

% max of UE power (Watt)
pt_UEk_max_lin = 10^(pt_UEk_max/10)*1e-3   ;

% Noise power calculation
temperature = 300;
Boltzmanns_constant = 1.38*10^-23;
noise_figure_dB = 8;
noise_figure_linear = 10^(noise_figure_dB./10);
subcarrier_bw = 15e+3  ;                             %KHz
thermal_noise = subcarrier_bw*temperature*Boltzmanns_constant*noise_figure_linear ;  % Watt

% SVD decomposition and CNR (channel to noise ratio), Eq.28
gamma_BS_S = [];
for c = 1:no_subcarriers
    
    % SVD decomposition for each subcarrier's channel
    [U_BS_S,S(:,:,c),V_BS_S] = svd(squeeze(h_BS_S(:,:,c)));
    sig_BS_S(:,c) = diag(S(:,:,c));
end
    % outdoor-to-indoor channel CNR for each subcarrier, \Lambda_B_S in Eq.28
    gamma_BS_S = sig_BS_S.*conj(sig_BS_S)./thermal_noise ;

% Number of Parallel links (this is important if in M*N mimo, M is not equal to N)
no_par_links = size(sig_BS_S,1);
    
% indoor channel CNR, \Lambda_S_UE in Eq.28
gamma_S_UE = h_S_UE.*conj(h_S_UE)./thermal_noise ;
% channel sorting (is possible because of joint processing between URUs) 
gamma_S_UE = sort(gamma_S_UE,2,'descend');
% Choosing the bests (if the number of URUs sim more than number of TX antennas, extra URUs will be autamtically turned off)
gamma_S_UE = gamma_S_UE(:,1:no_par_links,:);


% Uplink and Downlink both happen in same coherence time so CNRs are same,
% Eq.29 (This will be used if the uplink is also to be activated)
gamma_S_BS = gamma_BS_S' ;
gamma_UE_S = gamma_S_UE ;

gamma_BS_S = permute(repmat(gamma_BS_S,1,1,no_UE),[3 1 2]);
gamma_S_BS = permute(repmat(gamma_S_BS,1,1,no_UE),[3 2 1]);
%% Algorithm implementation (Algorithm 2, Alternating optimization)

no_iterarions = 20;

% Downlink matrices
P_BS_S_opt = [];                                              % BS_S power matrix
P_S_UE_opt = [];

% URUs_UE power intialization (we consider two cases, random initialization and equal power distribution)
% xx=rand(8,1);
% P_S_UE_opt(:,:,1) = (pt_URU_max_DLorUL_lin/sum(xx))*xx;  
P_S_UE_opt(:,:,:,1) = (pt_URU_max_DLorUL_lin/no_URUs)/no_subcarriers*ones(no_UE,no_par_links,no_subcarriers);


%Uplink Matrices
P_S_BS_opt = [];                                              % S_BS power matrix
% xx=rand(8,1);

% URUS_BS power intialization (we consider two cases, random initialization and equal power distribution)
% P_S_BS_opt(:,:,1) = (pt_URU_max_DLorUL_lin/sum(xx))*xx;   
P_S_BS_opt(:,:,:,1) = (pt_URU_max_DLorUL_lin/no_URUs)/no_subcarriers*ones(no_UE,no_par_links,no_subcarriers);
P_UE_S_opt = [] ;                                             % UE_S power matrix

% rate constraint (This part does not play any role in this code since "minimum rate" constraint is not considered, it should be kept equal to 0)
wDd = 0 ;                                                     % Langrangian multiplier for DL Minimum Rate constraint
wDu = 0 ;                                                    % Langrangian multiplier for UL Minimum Rate constraint

% Efficiency Coefficients (This part does not play any role in this code since "efficiency" is not considered, it should be kepts as it is)
neff = 0 ;
epB = 1 ;
epS = 1 ;

% k users
k = no_UE;                                                       

% This part does not play any role since "time sharing" constraint is not considered, it should be kept as  
s_DL = 1./no_UE*ones(no_UE,no_par_links,no_subcarriers);                         % DL transmission time portion
s_UL = 1./no_UE*ones(no_UE,no_par_links,no_subcarriers);                        % UL transmission time portion
S_DL_temp = [];

for h = 2:no_iterarions
    
    %*** Please note that all the powers calculated here are ACTUAL powers
    %(not effective powers), for obtaining the effective power we have to
    %consider the portion of time in which actual powers are  applied.
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%      DOWNLINK     %%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
   
    % Updating Langrangian Multiplier for Maximum BS DL power, Eq.68 (using BiSection
    % Method) 
    
    
    Lambda = 1e-15;                     % Lagrangian multiplier (initialization)
    stepsize = 1 ;                      % step size for finding the proper lagrangian multiplier
    temp_grad_dec = 0;                  
    P_BS_S_opt_temp = [] ;
    P_gamma_indoor_DL = P_S_UE_opt(:,:,:,h-1) .* gamma_S_UE ;
    while Lambda < 1e+6
        % Eq.68
        temp_BS_S = ((4*gamma_BS_S.*(wDd+1))./(gamma_S_UE.*P_S_UE_opt(:,:,:,h-1).*0.69314*(Lambda+neff*epB)));
        P_BS_S_opt_temp  = s_DL.* 1./(gamma_BS_S ) .* max(0, (0.5*P_gamma_indoor_DL.*(sqrt(1+temp_BS_S)-1)-1));
        
        % C1 in Eq.44
        tol = sum(sum(sum(P_BS_S_opt_temp)))-pt_max_BS_lin;
        
        % Checking whether the constraint is met        
        if tol < temp_grad_dec & tol < 0
            if abs(temp_grad_dec-tol) > pt_max_BS_lin/1000
                Lambda = Lambda - stepsize ;
                stepsize = stepsize/2;
                tol = temp_grad_dec;
            else
                P_BS_S_opt(:,:,:,h)=P_BS_S_opt_temp;
                break
            end
        else
            stepsize=2*stepsize;
        end
        temp_grad_dec = tol ;
        Lambda = Lambda + stepsize ;
    end
    if Lambda>= 1e+6
        disp('Constraint P_BS_S_opt CONNOT be met');
    end
    
    % Updating Langrangian Multiplier for Maximum URUs DL power, Eq.69 (using BiSection
    % Method) 
    
    Delta = 1e-15;                      % Lagrangian multiplier (initialization)
    stepsize = 1 ;                      % step size for finding the proper lagrangian multiplier
    temp_grad_dec =0;
    P_S_UE_opt_temp = [] ;
    P_gamma_O2I = P_BS_S_opt(:,:,:,h) .* gamma_BS_S ;
    
    while Delta < 1e+6
        % Eq.69
        temp_S_UE = ((4*gamma_S_UE.*(wDd+1))./(gamma_BS_S.*P_BS_S_opt(:,:,:,h).*0.69314*(Delta+neff*epB)));
        P_S_UE_opt_temp  = s_DL.* 1./(gamma_S_UE) .* max(0, (0.5*P_gamma_O2I.*(sqrt(1+temp_S_UE)-1)-1));
        
        % C2 in Eq.44
        tol = sum(sum(sum(P_S_UE_opt_temp)))- pt_URU_max_DLorUL_lin;
        
        % Checking whether the constraint is met (Bisection method)
        if tol < temp_grad_dec & tol < 0
            if abs(temp_grad_dec-tol) > pt_URU_max_DLorUL_lin/1000
                Delta = Delta - stepsize ;
                stepsize = stepsize/2;
                tol = temp_grad_dec;
            else
                P_S_UE_opt(:,:,:,h)=P_S_UE_opt_temp;
                break
            end
        else
            stepsize=2*stepsize;
        end
        temp_grad_dec = tol ;
        Delta = Delta + stepsize ;
    end
    if Delta>= 1e+6
        disp('Constraint P_S_UE_opt CONNOT be met');
    end
    
    
    
    
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%      UPLINK      %%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    
       
    % Updating Langrangian Multiplier for Maximum UE UL power, Eq.71 (using BiSection
    % Method) 
    
    Psi = 1e-15;                            % Lagrangian multiplier (initialization)
    stepsize = 1 ;                          % step size for finding the proper lagrangian multiplier
    temp_grad_dec = 0;
    P_UE_S_opt_temp = [] ;
    P_gamma_I2O = P_S_BS_opt(:,:,:,h-1) .* gamma_S_BS ;
    
    while Psi < 1e+6
        % Eq.71
        temp_UE_S = ((4*gamma_UE_S).*(wDu+1)./(gamma_S_BS.*P_S_BS_opt(:,:,:,h-1).*0.69314.*(Psi+neff.*epS)));
        P_UE_S_opt_temp = s_UL.*1./(gamma_UE_S) .*max(0, (0.5*P_gamma_I2O.*(sqrt(1+temp_UE_S)-1)-1) );
        
        % C3 in Eq.44
        tol = sum(sum(sum(P_UE_S_opt_temp)))- pt_UEk_max_lin;
        
        % Checking whether the constraint is met (Bisection method)
        if tol < temp_grad_dec & tol < 0 & temp_grad_dec ~=0
            if abs(temp_grad_dec-tol) > pt_UEk_max_lin/1000
                Psi = Psi - stepsize ;
                stepsize = stepsize/2;
                tol = temp_grad_dec;
            else
                P_UE_S_opt(:,:,:,h) = P_UE_S_opt_temp;
                break
            end
        else
            stepsize=2*stepsize;
        end
        Psi = Psi + stepsize ;
        temp_grad_dec = tol ;
    end
    if Psi>= 1e+6
        disp('Constraint P_UE_S_opt CONNOT be met');
    end
        
    
    % Updating Langrangian Multiplier for Maximum URUs UL power, Eq.70 (using BiSection
    % Method) 
    
    Phi = 1e-15;                        % Lagrangian multiplier (initialization)
    stepsize = 1 ;                      % step size for finding the proper lagrangian multiplier
    temp_grad_dec = 0;
    P_S_BS_opt_temp = [] ;
    P_gamma_indoor_UL = P_UE_S_opt(:,:,:,h) .* gamma_UE_S ;
    
    while Phi < 1e+6
        % Eq.70
        temp_S_BS = ((4*gamma_S_BS.*(wDu+1))./(gamma_UE_S.*P_UE_S_opt(:,:,:,h).*0.69314*(Phi+neff*epB)));
        P_S_BS_opt_temp =  s_UL .*1./(gamma_S_BS ) .* max(0, (0.5*P_gamma_indoor_UL.*(sqrt(1+temp_S_BS)-1)-1));
        
        % C4 in Eq.44
        tol = sum(sum(sum(P_S_BS_opt_temp)))-pt_URU_max_DLorUL_lin;
        
        % Checking whether the constraint is met
        if tol < temp_grad_dec & tol < 0 & temp_grad_dec ~= 0
            if abs(temp_grad_dec-tol) > pt_URU_max_DLorUL_lin/1000
                Phi = Phi - stepsize ;
                stepsize = stepsize/2;
                tol = temp_grad_dec;
            else
                P_S_BS_opt(:,:,:,h) = P_S_BS_opt_temp;
                break
            end
        else
            stepsize=2*stepsize;
        end
        temp_grad_dec = tol ;
        Phi = Phi + stepsize ;
    end
    if Phi >= 1e+6
        disp('Constraint P_S_BS_opt CONNOT be met');
    end
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%%%%%% Time sharing update within coherence time %%%%%%%%%%%%%%%%
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    
%     Rate_DL = [];
%     Rate_UL = [];
%     alpha = 0.04;   % minimum possible dedicated time
%     temp_grad_dec = 0;
%     counter = 1 ;
%     stepsize=0.01;
%         while counter<1e+6
%             temp_DL=0;
%             temp_UL=0;
%             for c = 1:no_subcarriers
%             temp_DL = temp_DL + alpha*subcarrier_bw*log2(det(eye(no_par_links)+diag(P_BS_S_opt(:,c,h)./alpha.*gamma_BS_S(:,c).*P_S_UE_opt(:,c,h)...
%                 .*gamma_S_UE(:,c)./(alpha.*ones(no_par_links,1)+P_BS_S_opt(:,c,h).*gamma_BS_S(:,c)+P_S_UE_opt(:,c,h).*gamma_S_UE(:,c)))));
%             
%             temp_UL = temp_UL + (1-alpha)*subcarrier_bw*log2(det(eye(no_par_links)+diag(P_S_BS_opt(:,c,h)./(1-alpha).*gamma_S_BS(:,c)...
%                 .*P_UE_S_opt(:,c,h).*gamma_UE_S(:,c)./((1-alpha).*ones(no_par_links,1)+P_S_BS_opt(:,c,h).*gamma_S_BS(:,c)+P_UE_S_opt(:,c,h)...
%                 .*gamma_UE_S(:,c)))));
%             end
%             temp = temp_DL + temp_UL ;
%             
%             if temp<temp_grad_dec || alpha > 0.96 
%                S_DL_temp(1,h) = alpha-stepsize ; 
%                break  
%             end
%             
%             alpha = alpha + stepsize;
%             temp_grad_dec = temp ;
%             counter = counter+1;
%         end
        
              S_DL_temp(1,h) = 1-1e-15;    % If you want to deActivate time
                                           % sharing --> deactivate the
                                           % upper part and activate
                                           % this line
              SINR = (gamma_BS_S.*P_BS_S_opt(:,:,:,h).* P_S_UE_opt(:,:,:,h)./s_DL.* gamma_S_UE)./...
                  (s_DL + gamma_BS_S.*P_BS_S_opt(:,:,:,h) + P_S_UE_opt(:,:,:,h).* gamma_S_UE);
              M_DL = sum((log2(1+SINR)-SINR./(1+SINR)),2);                             
              temp = max(M_DL,[],1);
              s_DL = zeros(no_UE,no_par_links,no_subcarriers);
              for l = 1:no_UE
                  logic0 = (M_DL(l,:,:)==temp)+ 1e-6;
                  s_DL(l,:,:) = repmat(logic0,no_par_links,1);
              end
    % Checking the convergence
    logic1 = (abs(P_BS_S_opt(:,:,:,h)-P_BS_S_opt(:,:,:,h-1))<= 1e-5);
    logic2 = (abs(P_S_UE_opt(:,:,:,h)-P_S_UE_opt(:,:,:,h-1))<= 1e-6);
    logic3 = (abs(P_UE_S_opt(:,:,:,h)-P_UE_S_opt(:,:,:,h-1))<= 1e-4);
    logic4 = (abs(P_S_BS_opt(:,:,:,h)-P_S_BS_opt(:,:,:,h-1))<=1e-4);
    logic5 = (abs(S_DL_temp(1,h)-S_DL_temp(1,h-1))<=1e-2);
    
    if all(logic1(:)==1) && all(logic2(:)==1) && all(logic3(:)==1) && all(logic4(:)==1) && all(logic5(:)==1)
        
        break
    end
    
end

% Returning the outputs
Y_P_BS_S_opt = P_BS_S_opt(:,:,:,h);
Y_P_S_UE_opt = P_S_UE_opt(:,:,:,h);
Y_P_UE_S_opt = P_UE_S_opt(:,:,:,h);
Y_P_S_BS_opt = P_S_BS_opt(:,:,:,h);
Y_s_DL = s_DL;

% sumRate Calculation, Eq.4

        Rate_DL = Y_s_DL.*subcarrier_bw.*(log2(1 + Y_P_BS_S_opt./Y_s_DL.*gamma_BS_S.*Y_P_S_UE_opt...
        .*gamma_S_UE./(Y_s_DL + Y_P_BS_S_opt.*gamma_BS_S+Y_P_S_UE_opt.*gamma_S_UE)));
    
         sumRate = sum(Rate_DL(:)) ;
         
                    % Cut-sett bound calculation
           
          Rate_cutset = Y_s_DL.*15e+3.*(log2(1 + Y_P_BS_S_opt./Y_s_DL.*gamma_BS_S));
    
          rate_cutset = sum(Rate_cutset(:)) ;
          
          sumRate = [sumRate;rate_cutset]; 
 
         

