function [Davg] = delayCalculator(T,priority, durationMatrix, iSymbol)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
%                         delayCalculator -function-
% ----------------------------------------------------------------------- %
% PROGRAM INTRODUCTION
% This program calculates the average delay for a given priority case. This
% is used in 'timingDiagrams' function, please see there. 
% ----------------------------------------------------------------------- %
% INPUTS 
% >>> T:                Transmission duration of base station for each 
%                           symbol [in seconds]
% >>> priority:         Priority matrix. It has the form: 
%
%                            |  
%                            |    
%                            |   
%                      UE(x) |  
%                            | 
%                            | _ _ _ _ _ _ URU(y)
%
% Note that it is 2D since we also have iSymbol (denoting which symbol we
% are dealing with) as input.
% 
% Example (Say M = 2, K = 3, and we want this kind of a transmission:
% 1-URU1 transmits to UE3
% 2-URU2 transmits to UE2
% 3-URU1 transmits to UE1
% 4-URU1 transmits to UE2
% 5-URU2 transmits to UE1
% 6-URU2 transmits to UE3
%                           --Matrix--
%                           | 3    5 |   
%                           | 4    2 |  
%                           | 1    6 | 
%                           --Matrix--
%
% >>> durationMatrix:   It has the form: 
% 
%                            |  
%                            |    
%                            |   
%                      UE(x) |  
%                            | 
%                            | _ _ _ _ _ _ URU(y)
%
% Since we have iSymbol, we do not have third dimension (see timingDiagrams
% function) 
% >>> iSymbol:          It says to this function with which symbol we are 
%                                                                  dealing.
% NOTE: By having the input iSymbol (index of symbol), we do deal with 2D
% matrices separately. These are processed and made 3D in timingDiagrams
% function. 
% OUTPUTS
% >>> Davg:    "Average delay" [seconds]
% ----------------------------------------------------------------------- %
% Author:       Berkan Kilic / kilicberkan95@gmail.com
% Supervisor:   Aravindh Krishnamoorthy / aravindh.krishnamoorthy@fau.de
% Institute:    Fraunhofer IIS, Erlangen/Germany
% Last Update:  10.08.2017 - 3.45 pm / by Berkan Kilic
% Comment:      Checked. Works properly at the last updated date. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% INITIALIZATION
M = size(durationMatrix,2);
K = size(durationMatrix,1);
D = zeros(1,K);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FORM DURATION MATRIX FOR THE DESIRED SYMBOL
durationMatrixReshaped = reshape(durationMatrix(:,:,iSymbol)', 1, M*K);
priorityMatrixReshaped = reshape(priority', 1, M*K);
% ----------------------------------------------------------------------- %


% ----------------------------------------------------------------------- %
% IMPLEMENTATION OF THE ALGORITHM TO CALCULATE DELAY
durationVector = zeros(1, M*K);
for i = 1:M*K
    durationVector(priorityMatrixReshaped(i)) = durationMatrixReshaped(i);
end

for i = 1:K
    for l = 1:max(priority(i,:))
        D(i) = D(i) + durationVector(l);
    end
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% TAKE 'MEAN' SINCE WE WANT AVERAGE DELAY
Davg = T + T*mean(D);
% ----------------------------------------------------------------------- %


