%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
%                       drawTimingDiagrams -script-
% ----------------------------------------------------------------------- %
% PROGRAM INTRODUCTION
% This program creates 3D durationMatrix, meaning that it collects
% durationMatrix elements for each symbol seperately. Randomness comes from
% some part of the script SUDAS_Layout. Also it collects sum rates for each
% symbol separately and creates a sum rate array to show it on the plot.
% Without this script, durationMatrix has the form (this durationMatrix is
% directly taken from the function 'opt_joint' in SUDAS_Layout script.
% ----------------------------------------------------------------------- %
% Even if this is not a function, some numbers should be properly defined
% in workspace to be able to run this script. I will call inputs and
% outputs of the script. 
% INPUTS 
% >>> B:        Number of Base Stations (B=1 for now)
% >>> T:        Transmission duration of base station for each symbol 
%                   [in seconds]
% >>> nSymbol:  Number of symbols to be transmitted
% >>> durationMatrix: 
%       |  
%       |    
%       |   
% UE(x) |  
%       | 
%       |_ _ _ _ _ _ URU(y)
%
% This program calls the function 'opt_joint' and forms different
% durationMatrix for different symbols. 
% OUTPUTS
% Note that the durationMatrix3D has the form: 
% >>> durationMatrix3D: 
%
%       |  Symbol(z)
%       |    /
%       |   /
% UE(x) |  / 
%       | /
%       |/_ _ _ _ _ _ URU(y)
%
% durationMatrix3D is the input of timingDiagrams function since it also
% includes the information for different symbols. 
% >>> sumRateArray
% The same opt_joint function also outputs sumRate, which is normally a
% number. We again form a sumRateArray including the information for 
% different symbols. 
% ----------------------------------------------------------------------- %
% Author:       Berkan Kilic / kilicberkan95@gmail.com
% Supervisor:   Aravindh Krishnamoorthy / aravindh.krishnamoorthy@fau.de
% Institute:    Fraunhofer IIS, Erlangen/Germany
% Last Update:  10.08.2017 - 3.45 pm / by Berkan Kilic
% Comment:      Checked. Works properly at the last updated date. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% Note that in the SUDAS_Layout, B,T and nSymbol values should be defined 
% properly.
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% INITIALIZATION
% See timingDiagrams function to learn about details of durationMatrix3D
durationMatrix3D = zeros(K, M, nSymbol);

% Sum rates will be displayed on the figure. It is different for each
% symbol. Therefore we need sumRateArray with dimension [1xnSymbols] 
sumRateArray = zeros(1, nSymbol);
% ----------------------------------------------------------------------- %
% Configurations
% The LTE mode in which the whole system works
[BandWidth,no_subcarriers,fftLength] = LTEmode(lteMode);
% mm-Wave transmitter antenna gain
mmw_Tx_antenna_gain_dB = 0;
% Line of Sight (LOS) or non-LOS for each URU-to-UE case
LOS = ones(K,M);
% number of base stations (BSs)
NoTx = B ;
% number transmit antennas at BS
no_TxArrayElements = N ;
% number of URUs (which are considered
no_URUs = M ;
% it can be either 'ula8'(8 antenna
BS_antenna_model = 'ula8' ;
% the general scenario in which we are working. In order to change it
% refer to Quadriga documentation, page 55.
GeneralScenario = 'WINNER_UMa2Indoor_C4_NLOS' ;
% Maximum BS transmission power (dBm)
pt_max = 46 ;
% max of URUs power (13 dBm per MHz)
pt_URU_max_DLorUL = 10.5 + 10*log10(BandWidth/1e+6) ;
% max of UE power dBm (13 dBm per MHz)
pt_UEk_max = 10*log10(BandWidth/1e+6) + 10.5 ;

% ----------------------------------------------------------------------- %
% FORM durationMatrix3D % sumRateArray
for i = 1:nSymbol
    % ------------------------------------------------------------------- %
    % The part creating randomness
    % Taken directly from sim_parameters script!
    RxTxDistance = randi([2 a],K,M);
    % Channel coefficients generation for all subcarriers
    h_BS_S = channel_O2I( BandWidth,no_subcarriers,CenterFrequency,NoTx,BS_antenna_model,...
        no_TxArrayElements,no_URUs,a,D,GeneralScenario ) ;
    h_S_UE = channel_Indoor_mmWave(IndoorScenario,no_subcarriers,...
        fftLength,M,K,RxTxDistance,mmw_Tx_antenna_gain_dB,LOS);
    [Y_P_BS_S_opt, Y_P_S_UE_opt, Y_P_UE_S_opt, Y_P_S_BS_opt, Y_s_DL,...
        sumRate] = opt_joint(h_BS_S,h_S_UE,pt_max,pt_URU_max_DLorUL,...
            no_subcarriers,M,K,pt_UEk_max);
    % Convert the power matrix into a duration matrix in a non-standard
    % way. Note that this approach does not yield optimal time fractions,
    % however, is a good demonstrator of the timing diagram simulations.
    durationMatrix = sum(Y_P_S_UE_opt,3)/sum(sum(sum(Y_P_S_UE_opt)));
    % ------------------------------------------------------------------- %
    
    % Take sum rate for each symbol
    % We divide it by 10^6 because we will display it on the figure in the
    % form of Mbps (mega = 10^6). 
    sumRateArray(i) = sum(sumRate)/10^6;
    
    % Create durationMatrix3D which includes the information for different
    % symbols. (Please see timingDiagrams function for details) 
    durationMatrix3D(:,:,i) = durationMatrix;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% CALL timingDiagrams FUNCTION TO SHOW PLOT
% Note that function takes durationMatrix3D, not durationMatrix! Reason is
% that timingDiagrams function is arranged for 3D matrices including the
% information about different symbols. (Dimensions include the information
% for different UEs, different URUs and different symbols, forming 3D
% matrix) 
[BS, URU, UE, deltaT] = timingDiagrams(B, T,...
                                    durationMatrix3D, sumRateArray);
% ----------------------------------------------------------------------- %

