function [bestOptimizedPriority, fixedOptimizedPriority] =...
                                        optimizer(durationMatrix, iSymbol)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
%                           optimizer -function- 
% ----------------------------------------------------------------------- %
% PROGRAM INTRODUCTION
% This program achieves the optimized priority matrices by following an
% algorithm proved on paper. 
% ----------------------------------------------------------------------- %
% INPUTS 
%
% >>> durationMatrix:   It has the form: 
% 
%                            |  
%                            |    
%                            |   
%                      UE(x) |  
%                            | 
%                            | _ _ _ _ _ _ URU(y)
%
% Since we have iSymbol, we do not have third dimension (see timingDiagrams
%                                                               function) 
% >>> iSymbol:          It says to this function with which symbol we are 
%                                                                  dealing.
% NOTE: By having the input iSymbol (index of symbol), we do deal with 2D
% matrices separately. These are processed and made 3D in timingDiagrams
% function. 
%
% OUTPUTS
% >>> bestOptimizedPriority:    This priority matrix is the best-optimized
% one. It does not have any constraints like fixedOptimizedPriority matrix.
% >>> fixedOptimizedPriority:  For this optimized priority matrix, our
% constraint is "not to turn off a relay until it transmits to every user
% equipment". In other words, if URU1 transmits to UE1 first, URU2 and the
% other relays do not start transmission until URU1 transmits to every UE
% separately. 
%
% Both of these priority matrices have the following form: 
% 
%                            |  
%                            |    
%                            |   
%                      UE(x) |  
%                            | 
%                            | _ _ _ _ _ _ URU(y)
% ----------------------------------------------------------------------- %
% Author:       Berkan Kilic / kilicberkan95@gmail.com
% Supervisor:   Aravindh Krishnamoorthy / aravindh.krishnamoorthy@fau.de
% Institute:    Fraunhofer IIS, Erlangen/Germany
% Last Update:  29.08.2017 - 10.02 am / by Berkan Kilic
% Comment:      Method to priority matrix giving the latency of Davg-fixed
%               is changed. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% EXTRACT M & K
% M: Number of relays
M = size(durationMatrix,2);

% K: Number of user equipments
K = size(durationMatrix,1);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% SUM OF ALL ELEMENTS IN EACH ROW & COLUMN 
rowSum = sum(durationMatrix(:,:,iSymbol),2);
rowSum = rowSum';
colSum = sum(durationMatrix(:,:,iSymbol),1);
colSum = colSum';
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% INITIALIZATION
indexVectorAvg = zeros(1,K);
indexVectorFixed = zeros(1,M);
indexVectorLastCol = zeros(1,K);
bestOptimizedPriority = zeros(K,M);
fixedOptimizedPriority = zeros(K,M);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% SORT IN DESCENDING FASHION
sortedRowSum = sort(rowSum);
sortedColSum = sort(colSum);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% ALGORITHM TO FIND bestOptimizedPriority 
for i =1:K
    indexVectorAvg(i) = find(rowSum(i) == sortedRowSum,1);
    % To cancel the selected term, we equate it to -1, which is an
    % impossible sum of relative durations
    sortedRowSum(indexVectorAvg(i)) = -1;
    bestOptimizedPriority(i,:) = M*(indexVectorAvg(i)-1)+1:...
                                            M*indexVectorAvg(i);
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% ALGORITHM TO FIND fixedOptimizedPriority
% For Davg-fixed, we have to take into account the column having greatest
% sum!
iLeastPriorColumn = find(colSum == max(colSum));
lastCol = durationMatrix(:,iLeastPriorColumn,iSymbol);
lastCol = lastCol';
sortedLastCol = sort(lastCol);
for i =1:M
    indexVectorFixed(i) = find(colSum(i) == sortedColSum,1);
    % To cancel the selected term, we equate it to -1, which is an
    % impossible sum of relative durations
    sortedColSum(indexVectorFixed(i)) = -1;
    fixedOptimizedPriority(:,i) = K*(indexVectorFixed(i)-1)+1:...
                                                K*indexVectorFixed(i);
    if i == iLeastPriorColumn
        for k = 1:K
            indexVectorLastCol(k) = find(lastCol(k) == sortedLastCol,1);
            sortedLastCol(indexVectorLastCol(k)) = -1;
            fixedOptimizedPriority(k,i) = (M-1)*K + indexVectorLastCol(k);
        end
    end
end
% ----------------------------------------------------------------------- %

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%OLD PARTS%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MAY BE USED FOR DEBUGGING
% Note that it is a very inefficient way to optimize. This code tries every
% possibility and chooses the one giving best results. We only used that to
% compare the results between this and the mathematical algorithm written 
% above. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%PRACTICAL OPTIMIZATION%%%%%%%%%%%%%%%%%%%%%%%%
% minVector = zeros(1, factorial(M*K));
% avgVector = zeros(1, factorial(M*K));
% 
% allPossiblePriorityMatrices = perms(1:M*K);
% for i = 1:factorial(M*K)
%     priority = allPossiblePriorityMatrices(i,:);
%     priority = reshape(priority, [M K])';
%     [Dmin, Davg] = delayCalculator(T,priority, durationMatrix, iSymbol);
%     minVector(i) = Dmin;
%     avgVector(i) = Davg;
% end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%PRACTICAL OPTIMIZATION%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%FINDING DMIN%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% index = find(rowSum == min(rowSum),1);
% minOptimizedPriorityMathematical(index,:) = 1:M;
% priorityOthers = reshape(M+1:M*K, M, K-1)';
% if index == 1
%     minOptimizedPriorityMathematical((2:K),:) = priorityOthers;
% elseif index == K
%     minOptimizedPriorityMathematical((1:K-1),:) = priorityOthers;
% else
%     minOptimizedPriorityMathematical([1:index-1,index+1:K],:) =...
%                                                   priorityOthers;
% end
%
% minDelay = min(minVector);
% minAvgDelay = min(avgVector);
%
% iOptimizedDmin = find(minVector==minDelay, 1, 'last');
% iOptimizedDavg = find(avgVector==minAvgDelay, 1, 'last');
%
% minOptimizedPriority =...
%          reshape(allPossiblePriorityMatrices(iOptimizedDmin, :), [M K])';
% bestOptimizedPriority =...
%          reshape(allPossiblePriorityMatrices(iOptimizedDavg, :), [M K])';
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%FINDING DMIN%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%OLD STAFF%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

