function [BS, URU, UE, deltaT] = timingDiagrams(B, T,...
                                            durationMatrix, sumRateArray)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
%                       timingDiagrams -function-
% ----------------------------------------------------------------------- %
% PROGRAM INTRODUCTION
% This program plots the timing diagrams of a SUDAS system. It shows the
% transmission, reception and processing signals of the elements used in
% SUDAS project, which are base station(s), relay(s), user equipment(s). 
% ----------------------------------------------------------------------- %
% INPUTS
% >>> B:              Number of Base Stations 
% >>> T:              Transmission duration of base station for each symbol 
%                                                              [in seconds]
% >>> durationMatrix: Relative duration matrix (3 dimensional) 
%                     Input should be in the following form: 
%
%       |  Symbol(z)
%       |    /
%       |   /
% UE(x) |  / 
%       | /
%       |/_ _ _ _ _ _ URU(y)
%
% For example, durationMatrix(3,2,1) * T gives time of "URU2 transmitting
% symbol1 to UE3"
% ----------------------------------------------------------------------- %
% Sample durationMatrix can be defined as (say 3 URUs, 3 UEs, 2 symbols): 
% durationMatrix = [0.1 0.1 0.2; 0.2 0.3 0.5; 0.6 0.3 0.7];
% durationMatrix(:,:,2) = [0.30 0.20 0.10; 0.50 0.50 0.70; 0.10 0.40 0.20];
% durationMatrixToUse(:,:,2) =...
%        durationMatrix(:,:,2)/(sum(sum(durationMatrix(:,:,2))));
% durationMatrixToUse(:,:,1) =...
%        durationMatrix(:,:,1)/(sum(sum(durationMatrix(:,:,1))));
% ------------------------------IMPORTANT!!!----------------------------- % 
% NOTE: Normally, program in which this functions works gives
% durationMatrix input itself. I just wrote that if you want to test this
% function only, apart from the whole program.
% ------------------------------IMPORTANT!!!----------------------------- % 
% >>> sumRateArray:   Please see drawTimingDiagrams script.
% 
% NOTE: You can change B and T values (also number of relays (SUDACs),
% number of users and number of symbols to be transmitted from SUDAS_Layout
% script. 
%
% OUTPUTS
% >>> BS:       Base Station
% >>> URU:      Relay
% >>> UE:       User equipment
% >>> deltaT:   Time difference between the points (on the plot) selected 
%                                                                   by user
% NOTE: In this case program also asks user whether he/she
% wants to see the time difference or not. If he/she says no (enter 0) then
% program automatically arranges time difference = 0 and only shows plot.
%
% Elements of the outputs to be plotted (plotting is automatically done) 
% >>> >>> BS(i).TX:     Transmission signal of the ith base station
% >>> >>> URU(i).RX:    RX signal of the ith URU 
% >>> >>> URU(i).P:     Process signal of the ith URU
% >>> >>> URU(i).TX:    TX signal of the ith URU
% >>> >>> UE(i).RX:     RX signal of the ith UE 
% >>> >>> UE(i).P:      Process signal of the ith UE 
% ----------------------------------------------------------------------- %
%  
% ----------------------------------------------------------------------- %
% NOTES
% 1- Please make sure that you define durationMatrix in advance in the case
% that you only want to run this function without running the big program.
% (Big program should automatically give a proper input to this function) 
% 2- For now, program only works for the case B = 1.
% 3- Normally, program is designed for durationMatrix following some rules
% to prevent factors decreasing efficiency (relays transmit just after 
% they complete processing) Therefore timeEnd is changeable such that
% program does not collapse after taking a proper durationMatrix input. 
% Whole program always gives this proper durationMatrix as input to that
% function. However, if you want to test just this function and give some 
% random matrices as input, you should just change timeEnd to allocate 
% enough time for every element. Otherwise you will encounter the error 
% like: 
% >>>
% Undefined function or variable 'risingEdgeBS'. Error in timingDiagrams...
%    line(xxx) for k = 1:length(risingEdgeBS)
% >>>
% 4- Please read all of the comments to understand the logic of the
% program. They are carefully prepared for the sake of other coders! Also
% if you want to make some changes, follow the fashion I used. Take a look
% at the website: 
% https://www.ee.columbia.edu/~marios/matlab/MatlabStyle1p5.pdf
% It is a nice coding standard for Matlab. 
% 5- Function is called in "drawTimingDiagrams" script. 
% 6- To increase accuracy of the measurement, decrease stepSize.  
% ----------------------------------------------------------------------- %
% Author:       Berkan Kilic / kilicberkan95@gmail.com
% Supervisor:   Aravindh Krishnamoorthy / aravindh.krishnamoorthy@fau.de
% Institute:    Fraunhofer IIS, Erlangen/Germany
% Last Update:  10.08.2017 - 3.45 pm / by Berkan Kilic
% Comment:      Checked. Works properly at the last updated date. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
% --------------------INITIALIZATIONS & PRE-DEFINITIONS-------------------%
% ----------------------------------------------------------------------- %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% NOTE: USER MAY WANT TO CHANGE SOMETHING IN THAT SECTION
% e.g. priority matrix, process times and so on... 
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FIND NUMBER OF RELAYS, USER EQUIPMENTS, SYMBOLS
% See SUDAS_Layout script.
% K:        Number of Users
% M:        Number of SUDACs
% nSymbol:  Number of symbols
K = size(durationMatrix, 1);
M = size(durationMatrix, 2);
nSymbol = size(durationMatrix,3);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FIND NUMBER OF ELEMENTS IN THE PROGRAM
% Total number of all signals in the program
% (TX signal for BS) + (RX/Process/TX signals for URU) +...
% (RX/Process signals for UE)
% Note that 1 stands for TX (of BS),
% 3 stands for RX/Process/TX (of URUs), 
% 2 stands for RX/Process (of UEs),
nElement = B + (M*3) + (K*2);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% Transmission duration of the base station(s)
BS(1).TXduration =  T;
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% Process durations of the UEs
UE = struct('Pduration', num2cell(10^-6*ones(1, K)));
% This code is equivalent to the following: 
% for l =1:K
%     UE(l).Pduration = 10^-6;
% end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% Transmission & Process durations of the URUs
% Note that durationMatrix is used here to determine the transmission times
% ----------------------------IMPORTANT NOTE------------------------------
% In my program, I use the following convention: 
% URU(i,l,k) where i: index of URU, l; index of UE, k: index of symbol
% The durationMatrix input has the form UE in x axis, URU in y axis and
% symbols in z axis. Therefore, I just take the transpose of that matrix
% here and assing the TX durations accordingly. We do not have any other
% transpose or something after that point. Also note that original
% durationMatrix is kept for the rest of code. 
% ----------------------------IMPORTANT NOTE------------------------------
newDurationMatrix = zeros(M,K,nSymbol);
for k = 1:nSymbol
    newDurationMatrix(:,:,k) = durationMatrix(:,:,k)';
end

URU = struct('Pduration', num2cell(10^-6*ones(M, K, nSymbol)),...
    'TXduration', num2cell(T*newDurationMatrix((1:M),(1:K),(1:nSymbol))));
% This code is equivalent to the following: 
% for i = 1:M
%     for l = 1:K
%         for k = 1:nSymbol
%             URU(i,l,k).TXduration = T*newDurationMatrix(i,l,k);
%             URU(i,l,k).Pduration = 10^-6;
%         end
%     end
% end
% ----------------------------------------------------------------------- %

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
% PRIORITIES
% See delayCalculator function to learn the structure of priority matrix.
% Here we create 3D priority matrix having the information about symbols in
% its z-axis. It has the following form: 
%
%       |  Symbol(z)
%       |    /
%       |   /
% UE(x) |  / 
%       | /
%       |/_ _ _ _ _ _ URU(y)
% ----------------------------------------------------------------------- %
%
% ----------------------------------------------------------------------- %
% INITIALIZATION 
priority = zeros(K,M, nSymbol);
minAvgDelay = zeros(1,nSymbol);
fixedAvgDelay = zeros(1,nSymbol);
% ----------------------------------------------------------------------- %
%
% ----------------------------------------------------------------------- %
% INSERT THE SYMBOL INFORMATION
% See delayCalculator & optimizer
for k = 1:nSymbol
    [bestOptimizedPriority, fixedOptimizedPriority] =...
                            optimizer(durationMatrix, k);
    % Priority is used to plot. It is adjusted to best scenario in terms of
    % delays. If you want to see other case, change here to 
    % priority(:,:,k) = fixedOptimizedPriority 
    priority(:,:,k) = bestOptimizedPriority;
    minAvgDelay(k) =...
            delayCalculator(T,bestOptimizedPriority, durationMatrix, k);
    fixedAvgDelay(k) =...
            delayCalculator(T,fixedOptimizedPriority, durationMatrix, k);
end
% ----------------------------------------------------------------------- %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% TIME PROPERTIES
% Decrease stepSize to increase accuracy. Increase stepSize to increase
% speed
stepSize    = 10^-8;
timeEnd     = (nSymbol+3)*T;
t           = (0:stepSize:timeEnd);
% ----------------------------------------------------------------------- %

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
%-----------------------------SIGNAL FORMATIONS-------------------------- %
% ----------------------------------------------------------------------- %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% NOTE: If you are a user, then do not change anything in this section.
% However if you are a coder working on further improvements and/or
% debugging, this the section of the code is the one you should play with. 
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% INDEXING
% Since for-loop in Matlab is sample-based, index of the corresponding
% transmission signal is computed. This logic is followed in many parts of
% the program.
BS(1).TXindex = find(t > BS(1).TXduration, 1);
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% Forming the BS(1).TXsignal
BS(1).TXsignal(1, length(t)) = 0;
for k = 1:nSymbol
    BS(1).TXsignal((k-1)*BS(1).TXindex+k+1:k*BS(1).TXindex+k) = 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FALLING & RISING EDGES OF BS_TX SIGNAL
indexFalling = 1;
indexRising = 1;
s = BS(1).TXsignal;

% Pre-allocation
fallingEdgeBS = zeros(1,nSymbol);
risingEdgeBS = zeros(1,nSymbol);

for i = 1:length(s)-1
    if s(i) == 1 && s(i+1) == 0
        fallingEdgeBS(indexFalling) = i;
        indexFalling = indexFalling + 1;
    end
    if s(i) == 0 && s(i+1) == 1
        risingEdgeBS(indexRising) = i;
        indexRising = indexRising + 1;
    end
end    
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FORMING URU RX SIGNALS
URU_RX = struct('signal', num2cell(zeros(M,1)));
for i = 1:M
    URU_RX(i).signal = zeros(1,length(t));
    for k = 1:length(risingEdgeBS)
        URU_RX(i).signal(risingEdgeBS(k)+1:fallingEdgeBS(k)) = 1;
    end
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FORMING URU PROCESS SIGNALS
URU_P = struct('signal', num2cell(zeros(M,K)));
for i = 1:M
    for k = 1:nSymbol
        % Duration in index forms (not time!)
        duration = ceil(URU(i,1,k).Pduration/stepSize);
        URU_P(i,k).signal = zeros(1,length(t));
        URU_P(i,k).signal(fallingEdgeBS(k)+1:...
                            fallingEdgeBS(k)+duration) = 1;
    end
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FALLING & RISING EDGES OF PROCESS SIGNALS
% Pre-allocation
fallingEdgeP = zeros(M, nSymbol);
risingEdgeP = zeros(M, nSymbol);
for i = 1:M
    for k =1:nSymbol
    s = URU_P(i,k).signal;
        for l = 1:length(s)-1
            if s(l) == 1 && s(l+1) == 0
                fallingEdgeP(i,k) = l;
            end
            if s(l) == 0 && s(l+1) == 1
                risingEdgeP(i,k) = l;
            end
        end
    end
end  
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% FORMING TX SIGNALS
% First TX signal is created since it depends on the process signals
% 3rd dimension, zeros(1, length(t),nUserEquipment) indicates to which UE
% URU transmits. Other two dimensions {1, length(t)} determines the signal. 
URU_TX = struct('signal', num2cell(zeros(M,nSymbol)));
for i = 1:M
    for k = 1:nSymbol
        URU_TX(i,k).signal = zeros(1, length(t), K);
    end
end

% Pre-allocation
fallingEdgeTX = zeros(M*K, nSymbol);
risingEdgeTX = zeros(M*K, nSymbol);

% Temporary risingEdgeTX used for comparison 
% They are to defined to find whether process signal of the URU itself or 
% TX of other URU comes first
risingEdgeTX1 = zeros(M*K, nSymbol);
risingEdgeTX2 = zeros(M*K, nSymbol);
% Formation
for k = 1:nSymbol 
    for i=1:M*K
    indexMatrix = find(priority(:,:,k)==i);
    [row, col] = ind2sub(size(priority(:,:,k)),indexMatrix);
    iURU = col;
    iUE = row; 
        if i == 1 && k == 1
            duration = ceil(URU(iURU,iUE,1).TXduration/stepSize);
            risingEdgeTX(1,1) = fallingEdgeP(1,1);
            fallingEdgeTX(1, 1) = risingEdgeTX(1,1)+duration;
            URU_TX(iURU,k).signal(1,risingEdgeTX(1,1)+1:...
                                fallingEdgeTX(1,1),iUE) = 1;
        elseif i == 1 && k ~= 1
            duration = ceil(URU(iURU,iUE,k).TXduration/stepSize);
            risingEdgeTX1(i,k) = fallingEdgeTX(M*K,k-1)+1;
            risingEdgeTX2(i,k) = fallingEdgeP(i,k);
            risingEdgeTX = max(risingEdgeTX1, risingEdgeTX2);
            fallingEdgeTX(i,k) = risingEdgeTX(i,k)+duration;
            URU_TX(iURU,k).signal(1,risingEdgeTX(i,k)+1:...
                                    fallingEdgeTX(i,k),iUE) = 1;
        else
            duration = ceil(URU(iURU,iUE,k).TXduration/stepSize);
            risingEdgeTX(i,k) = fallingEdgeTX(i-1,k);
            fallingEdgeTX(i,k) = risingEdgeTX(i,k)+duration;
            URU_TX(iURU,k).signal(1,risingEdgeTX(i,k)+1:...
                                    fallingEdgeTX(i,k),iUE) = 1;
        end
    end
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% UE RX SIGNALS
for i = 1:K
    UE(i).RXsignal = zeros(1, length(t));
    for k = 1:M
        for l = 1:nSymbol
            UE(i).RXsignal = UE(i).RXsignal + URU_TX(k,l).signal(:,:,i);
        end
    end
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% UE-PROCESS SIGNALS
% Detection of falling&rising edges of UE-RX signals
fallingEdgeUE_RX = zeros(K, nSymbol*M);
risingEdgeUE_RX = zeros(K, nSymbol*M);
UE_P = struct('signal', num2cell(zeros(1,K)));
for i = 1:K
    UE_P(i).signal = zeros(1, length(t));
    indexFalling = 1;
    indexRising = 1;
    s = UE(i).RXsignal;
    for k =1:length(s)-1
        if s(k) == 1 && s(k+1) == 0 
            fallingEdgeUE_RX(i, indexFalling) = k;
            indexFalling = indexFalling + 1;
        end
        if s(k) == 0 && s(k+1) == 1
            risingEdgeUE_RX(i, indexRising) = k;
            indexRising = indexRising + 1;
        end
    end
end

% Forming UE_Process signals
for i = 1:K
    % Duration in index forms (not time!)
    duration = ceil(UE(i).Pduration/stepSize);
    for k = 1:nSymbol
        if fallingEdgeUE_RX(i, k) ~= 0
             UE_P(i).signal(fallingEdgeUE_RX(i,k)...
                 +1:fallingEdgeUE_RX(i,k)+duration) = 1;
        end
    end
end
% ----------------------------------------------------------------------- %

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ----------------------------------------------------------------------- %
%--------------------------------OUTPUTS--------------------------------- %
% ----------------------------------------------------------------------- %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ----------------------------------------------------------------------- %
% In this section, signals are modified such that they look better on plot.
% Plotting and user interaction commands are also in this section. 
% Modification of signals means that, for example, URU1 transmitting
% symbol1 to UE1 & URU1 transmitting symbol2 to UE1... are combined and one
% signal including all of these is created. 
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% INITIALIZATION
element(1:nElement, 1:length(t)) = 0;
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% BS SIGNAL
index = 1;
BS.TX = BS.TXsignal;
for i = 1:B
    element(index, 1:length(t)) = BS(i).TX;
    index = index + 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% URU-RX SIGNALS
for i = 1:M
    URU(i).RX = zeros(1,length(t));
    URU(i).RX = URU(i).RX+ URU_RX(i).signal;
    element(index, 1:length(t)) = URU(i).RX;
    index = index + 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% URU-PROCESS SIGNALS
for i = 1:M
    URU(i).P = zeros(1,length(t));
    for k = 1:nSymbol
        URU(i).P = URU(i).P + URU_P(i,k).signal;
    end
    element(index,1:length(t)) = URU(i).P;
    index = index + 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% URU-TX SIGNALS
% One signal including information for all symbols is created here. 
for i = 1:M
    URU(i).TX = zeros(1,length(t));
    for k =1:nSymbol
        for l =1:K
            URU(i).TX = URU(i).TX + URU_TX(i,k).signal(:,:,l);
        end
    end
    element(index,1:length(t)) = URU(i).TX;
    index = index + 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% UE-RX SIGNALS
for i = 1:K
    UE(i).RX = UE(i).RXsignal;
    element(index,1:length(t)) = UE(i).RX;
    index = index + 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% UE-PROCESS SIGNALS
for i = 1:K
    UE(i).P = UE_P(i).signal;
    element(index,1:length(t)) = UE(i).P;
    index = index + 1;
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% PLOT TITLES "AUTOMATIC ARRANGEMENT"
plotTitles = cell(1, nElement);
for i =1:B
    plotTitles{i} = ['BS' num2str(i) '-TX'];
end

for i = 1:M
    plotTitles{i+B} = ['URU' num2str(i) '-RX'];
end

for i = 1:M
    plotTitles{i+B+M} = ['URU' num2str(i) '-P'];
end

for i = 1:M
    plotTitles{i+B+M+M} = ['URU' num2str(i) '-TX'];
end

for i =1:K
    plotTitles{i+B+M+M+M} = ['UE' num2str(i) '-RX'];
end

for i =1:K
    plotTitles{i+B+M+M+M+K} = ['UE' num2str(i) '-P'];
end
% ----------------------------------------------------------------------- %

% ----------------------------------------------------------------------- %
% PLOTTING
% Rest of the code includes plotting. This part is long because we attached
% a great importance on the demonstration. 

% The following measurement code is for debugging and is commented out in
% the release version.
% if nargout == 4
%     userChoice = input(['Do you want to measure the time difference?'...
%         ' Enter 1 for YES and 0 for NO: ']);
%     if userChoice == 1
%         fprintf('--------------------------------------------------- \n');
%         fprintf('It seems you want to measure the time difference \n');
%         fprintf(['Just select 2 data points between which you'...
%                         ' want to measure the time difference \n']);
%         fprintf('To see the plot, press any key on your keyboard \n');
%         fprintf('--------------------------------------------------- \n');
%         pause
%     else
%         deltaT = 0;
%     end
% end

% Determine the locations where there is no transmission and denote them by
% red lines. To make user not to confuse, red lines are added to indicate 
% that "for these symbols, there in no transmission"
iSymbol = 0;
iUE = 0;
for i=1:K
    for k=1:nSymbol
        if durationMatrix(i,:,k) == 0
            iSymbol(end+1) = k;
            iUE(end+1) = i;
        end
    end
end
additionFactor = nElement - K;
iUE = iUE + additionFactor;
figure
plt = zeros(1,nElement)';
for i = 1:nElement
    plt(i) = subplot(nElement, 1, i);
    stairs(t, element(i, 1:length(t)), 'Color', 'Yellow');
    for index = 2:length(iUE)
        if i == iUE(index)
            line([(iSymbol(index)+1)*T-T/2 (iSymbol(index)+1)*T-T/2],...
                                                    [0 1], 'Color', 'Red');
        end
    end
    % gca.YTick([]);
    set(gca,'ylim',[-0.05 1.05]);
    fig = gcf;
    set(fig, 'units','normalized','outerposition',[0 0 1 1]); 
    fig.Color = [0.5 0.5 0.5];
    set(gca,'Color','Black');
    ylabel(plotTitles(i), 'FontSize', 8, 'Rotation', 30, 'Color', 'Green');
    ylabh = get(gca,'YLabel');
    set(ylabh,'Position',get(ylabh,'Position') - [0 0.4 0]);
    if i == 1
        text(0, 1.3+2*nElement*0.005, 'Davg-min (us): ', 'Color',...
                                                  'White', 'FontSize', 10);
        text(0, 1.6+6*nElement*0.005, 'Davg-fixed (us): ', 'Color',...
                                                  'White', 'FontSize', 10);
        for k =1:nSymbol
            text(T*(k+1)-T/3, 1.3+2*nElement*0.005,...
                    num2str(round(minAvgDelay(k)*10^6,2)), 'Color',...
                            'White', 'FontSize', 10);
            text(T*(k+1)-T/3, 1.6+6*nElement*0.005,...
                    num2str(round(fixedAvgDelay(k)*10^6,2)),...
                            'Color', 'White', 'FontSize', 10);
        end
    end
    if i > B + M*2
        for k = 1:nSymbol+1
            line([T*k T*k], [0 1], 'Color', 'Blue')
        end
    end
    if i == nElement
        text(0, -1-nElement*0.01, 'Sum Rate (Mbps)', 'Color',...
                                            'White', 'FontSize', 10);
        for k = 1:nSymbol
            text(T*(k+1)-T/3, -1-nElement*0.01,...
                        num2str(round((sumRateArray(k)),2)), 'Color',...
                                        'White', 'FontSize', 10);
        end
    else
        set(gca,'XTick',[]);
    end
end
set(zoom,'Motion','horizontal', 'Enable', 'on');
linkaxes(plt,'x');
deltaT = 0 ;
% The following measurement code is for debugging and is commented out in
% the release version.
% if nargout == 4 && userChoice == 1
%     fprintf('--------------------------------------------------- \n');
%     fprintf('Press any key to continue after choosing the values \n');
%     fprintf('--------------------------------------------------- \n');
%     check = 0;
%     while check == 0
%         dcm_obj = datacursormode(fig);
%         set(dcm_obj,'Enable','on');
%         pause
%         cursorInfo = getCursorInfo(dcm_obj);
%         if size(cursorInfo,2)==2
%             deltaT =...
%                 abs(cursorInfo(1).Position(1) - cursorInfo(2).Position(1));
%             check = 1;
%             fprintf(['------------------------'...
%                                 '--------------------------- \n']);
%             fprintf(['Time difference between the points selected: \n'...
%                                 num2str(deltaT) '\t\t\t(s)\n']);
%             deltaT_nano = deltaT*10^9;
%             deltaT_micro = deltaT*10^6;
%             deltaT_mili = deltaT*10^3;
%             fprintf([num2str(deltaT_mili) '\t\t\t\t(ms) \n']);
%             fprintf([num2str(deltaT_micro) '\t\t\t\t(us) \n']);
%             fprintf([num2str( deltaT_nano) '\t\t\t\t(ns) \n']);
%             fprintf(['------------------------'...
%                                     '--------------------------- \n']);
%         elseif size(cursorInfo,2) > 2
%             deltaT = 0;
%             fprintf(['---------------------------'...
%                                     '------------------------ \n']);
%             fprintf(['You pointed more than 2 values.'...
%                                 ' Please point exactly 2 values! \n']);
%             fprintf(['You can delete one of them'...
%                 ' by using -delete key on your keyboard \n']);
%             fprintf(['Press any key to continue'...
%                         ' after choosing the values properly \n']);
%             fprintf(['------------------------'...
%                                     '--------------------------- \n']);
%         elseif size(cursorInfo,2) == 0
%             deltaT = 0;
%             fprintf(['------------------------'...
%                         '--------------------------- \n']);
%             fprintf(['You did not point any values.'...
%                                 ' Please point exactly 2 values! \n']);
%             fprintf(['Just click the desired place on the figure.'...
%                         ' Cursor mode is already selected \n']);
%             fprintf(['To select more than 1 data point,'...
%                             ' use "alt+left click" \n']);
%             fprintf(['Press any key to continue after'...
%                             ' choosing the values properly \n']);
%             fprintf(['------------------------'...
%                                     '--------------------------- \n']);
%         else
%             deltaT = 0;
%             fprintf(['------------------------'...
%                                     '--------------------------- \n']);
%             fprintf(['You pointed only 1 value.'...
%                                  ' Please point exactly 2 values! \n']);
%             fprintf(['To select more than 1 data point,'...
%                                             ' use "alt+left click" \n']);
%             fprintf(['Press any key to continue after'...
%                                     ' choosing the values properly \n']);
%             fprintf(['------------------------'...
%                                         '--------------------------- \n']);
%         end
%     end
% end
% ----------------------------------------------------------------------- %
